import sys
import random
import time
import os
import json # 音量設定の保存用にインポート
from PyQt6.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QPushButton, QLabel,
    QStackedWidget, QMessageBox, QHBoxLayout, QGraphicsView,
    QGraphicsScene, QInputDialog, QTextEdit, QSlider, QFrame,
    QGraphicsTextItem, QGraphicsRectItem
)
from PyQt6.QtCore import Qt, QTimer, QRectF, QPointF, QPropertyAnimation,QPoint
from PyQt6.QtGui import QBrush, QColor, QPen, QPixmap, QPainter, QIcon, QKeySequence, QPainterPath

# オーディオ再生のためにpygameを使用
import pygame

def resource_path(relative_path):
    """ アセットへの絶対パスを生成します。 """
    if hasattr(sys, '_MEIPASS'):
        # PyInstallerによって作成された一時フォルダからパスを取得
        return os.path.join(sys._MEIPASS, relative_path)
    # 通常のPython環境では、実行中のカレントディレクトリを基準にする
    return os.path.join(os.path.abspath("."), relative_path)

# --- 定数 ---
BLOCK_SIZE = 24  # 1ブロックのピクセルサイズ
BASE_BOARD_WIDTH = 10 # ボードの幅（ブロック単位）
BOARD_HEIGHT = 20 # ボードの高さ（ブロック単位）

# --- ゲームモード ---
MODE_PVP = "1P vs 2P"
MODE_PVC = "1P vs CPU"

# --- アセットパス ---
TITLE_IMAGE_PATH = resource_path("tect/TetRival/Tetriatitle.png")
TITLE_BACKGROUND_IMAGE_PATH = resource_path("tect/TetRival/Tetriatitlewindow.png")
APP_ICON_PATH = resource_path("tect/TetRival/TetriaRivallogo.ico")
CREDITS_BACKGROUND_IMAGE_PATH = resource_path("tect/TetRival/img/TetriaRivalending.png")
GCS2_LOGO_PATH = resource_path("tect/TetRival/img/GCS2.png")
SAPOCREATE_LOGO_PATH = resource_path("tect/TetRival/img/SAPOCREATE.png")
GCOT_LOGO_PATH = resource_path("tect/TetRival/img/GCOT.png")

# --- BGM/SE パス ---
TITLE_BGM_PATH = resource_path("tect/TetRival/bgm/281_long_BPM100.mp3")
GAME_BGM_PATH = resource_path("tect/TetRival/bgm/243_long_BPM181.mp3")
CREDITS_BGM_PATH = resource_path("tect/TetRival/bgm/236_long_BPM87.mp3")
LINE_CLEAR_1_3_SE_PATH = resource_path("tect/TetRival/se/カーソル移動5.wav")
TETRIS_SE_PATH = resource_path("tect/TetRival/se/電子ルーレット停止ボタンを押す.wav")
CURSOR_MOVE_SE_PATH = resource_path("tect/TetRival/se/カーソル移動2.wav")
DECISION_SE_PATH = resource_path("tect/TetRival/se/決定ボタンを押す7.wav")


# --- 設定ファイルパス ---
KEY_CONFIG_FILE_RIVAL = resource_path("tect/TetRival/key_config_rival.txt")
VOLUME_CONFIG_FILE = resource_path("tect/TetRival/volume_config.json") # 音量設定ファイル

# --- デフォルトキーマッピング ---
DEFAULT_KEY_CONFIG = {
    'p1': {
        'left': Qt.Key.Key_A, 'right': Qt.Key.Key_D, 'down': Qt.Key.Key_S,
        'rotate_right': Qt.Key.Key_W, 'rotate_left': Qt.Key.Key_Q,
        'hard_drop': Qt.Key.Key_Shift, 'hold': Qt.Key.Key_C,
    },
    'p2': {
        'left': Qt.Key.Key_Left, 'right': Qt.Key.Key_Right, 'down': Qt.Key.Key_Down,
        'rotate_right': Qt.Key.Key_Up, 'rotate_left': Qt.Key.Key_M,
        'hard_drop': Qt.Key.Key_Control, 'hold': Qt.Key.Key_N,
    },
    'p1_cpu': {
        'left': Qt.Key.Key_Left, 'right': Qt.Key.Key_Right, 'down': Qt.Key.Key_Down,
        'rotate_right': Qt.Key.Key_Up, 'rotate_left': Qt.Key.Key_M,
        'hard_drop': Qt.Key.Key_Control, 'hold': Qt.Key.Key_N,
    }
}

# --- テトリミノの形と色 ---
STANDARD_TETROMINOS = {
    'I': [[[0,0,0,0], [1,1,1,1], [0,0,0,0], [0,0,0,0]], [[0,0,1,0], [0,0,1,0], [0,0,1,0], [0,0,1,0]]],
    'J': [[[1,0,0], [1,1,1], [0,0,0]], [[0,1,1], [0,1,0], [0,1,0]], [[0,0,0], [1,1,1], [0,0,1]], [[0,1,0], [0,1,0], [1,1,0]]],
    'L': [[[0,0,1], [1,1,1], [0,0,0]], [[0,1,0], [0,1,0], [0,1,1]], [[0,0,0], [1,1,1], [1,0,0]], [[1,1,0], [0,1,0], [0,1,0]]],
    'O': [[[1,1], [1,1]]],
    'S': [[[0,1,1], [1,1,0], [0,0,0]], [[0,1,0], [0,1,1], [0,0,1]]],
    'T': [[[0,1,0], [1,1,1], [0,0,0]], [[0,1,0], [0,1,1], [0,1,0]], [[0,0,0], [1,1,1], [0,1,0]], [[0,1,0], [1,1,0], [0,1,0]]],
    'Z': [[[1,1,0], [0,1,1], [0,0,0]], [[0,0,1], [0,1,1], [0,1,0]]]
}
TETROMINO_COLORS = {
    'I': QColor(0, 255, 255), 'J': QColor(0, 0, 255), 'L': QColor(255, 165, 0),
    'O': QColor(255, 255, 0), 'S': QColor(0, 255, 0), 'T': QColor(128, 0, 128),
    'Z': QColor(255, 0, 0), 'empty': QColor(30, 30, 40), 'ghost': QColor(255, 255, 255, 40),
    'garbage': QColor(130, 130, 130)
}

# --- ヘルパー関数 ---
def key_to_string(key_code):
    """Qt.Keyの整数値をその文字列表現に変換します。"""
    return QKeySequence(key_code).toString()

class TetrisAI:
    """CPUプレイヤーの思考ロジックを実装するクラス。"""
    def __init__(self, logic):
        self.logic = logic
        self.WEIGHT_HEIGHT = -4.5
        self.WEIGHT_LINES = 3.4
        self.WEIGHT_HOLES = -7.8
        self.WEIGHT_BUMPINESS = -1.8

    def find_best_move(self):
        if not self.logic.current_piece:
            return None
        best_score = -float('inf')
        best_move = None
        for piece_to_try in [self.logic.current_piece, self.logic.held_piece]:
            if piece_to_try is None:
                continue
            shape_key = piece_to_try['shape']
            num_rotations = len(STANDARD_TETROMINOS[shape_key])
            for r in range(num_rotations):
                piece_matrix = self._get_rotated_matrix(shape_key, r)
                if not piece_matrix: continue
                for x in range(-2, BASE_BOARD_WIDTH):
                    if self.logic._is_valid_position(piece_matrix, x, 0):
                        temp_board = [row[:] for row in self.logic.board]
                        y = self._get_landing_y(temp_board, piece_matrix, x)
                        self._place_piece_on_board(temp_board, piece_matrix, x, y, shape_key)
                        score = self._evaluate_board(temp_board)
                        if score > best_score:
                            best_score = score
                            needs_hold = (piece_to_try == self.logic.held_piece)
                            best_move = {'rotation': r, 'x': x, 'score': score, 'hold': needs_hold}
        return best_move

    def _get_rotated_matrix(self, shape_key, rotation):
        rotations = STANDARD_TETROMINOS[shape_key]
        return rotations[rotation % len(rotations)]

    def _get_landing_y(self, board, piece_matrix, x):
        y = 0
        while self._is_valid_on_temp_board(board, piece_matrix, x, y + 1):
            y += 1
        return y

    def _is_valid_on_temp_board(self, board, piece_matrix, x_offset, y_offset):
        for y, row in enumerate(piece_matrix):
            for x, cell in enumerate(row):
                if cell:
                    board_x, board_y = x + x_offset, y + y_offset
                    if not (0 <= board_x < BASE_BOARD_WIDTH and 0 <= board_y < BOARD_HEIGHT and board[board_y][board_x] == 'empty'):
                        return False
        return True

    def _place_piece_on_board(self, board, piece_matrix, x_offset, y_offset, shape_key):
        for y, row in enumerate(piece_matrix):
            for x, cell in enumerate(row):
                if cell:
                    if 0 <= y + y_offset < BOARD_HEIGHT and 0 <= x + x_offset < BASE_BOARD_WIDTH:
                        board[y + y_offset][x + x_offset] = shape_key
    
    def _evaluate_board(self, board):
        height = self._calculate_aggregate_height(board)
        lines = self._calculate_cleared_lines(board)
        holes = self._calculate_holes(board)
        bumpiness = self._calculate_bumpiness(board)
        score = (self.WEIGHT_HEIGHT * height + self.WEIGHT_LINES * lines + self.WEIGHT_HOLES * holes + self.WEIGHT_BUMPINESS * bumpiness)
        return score

    def _calculate_aggregate_height(self, board):
        height = 0
        for x in range(BASE_BOARD_WIDTH):
            for y in range(BOARD_HEIGHT):
                if board[y][x] != 'empty':
                    height += (BOARD_HEIGHT - y)
                    break
        return height

    def _calculate_cleared_lines(self, board):
        lines = 0
        for y in range(BOARD_HEIGHT):
            if all(cell != 'empty' for cell in board[y]):
                lines += 1
        return lines

    def _calculate_holes(self, board):
        holes = 0
        for x in range(BASE_BOARD_WIDTH):
            block_found = False
            for y in range(BOARD_HEIGHT):
                if board[y][x] != 'empty':
                    block_found = True
                elif block_found and board[y][x] == 'empty':
                    holes += 1
        return holes

    def _calculate_bumpiness(self, board):
        bumpiness = 0
        heights = []
        for x in range(BASE_BOARD_WIDTH):
            col_height = 0
            for y in range(BOARD_HEIGHT):
                if board[y][x] != 'empty':
                    col_height = BOARD_HEIGHT - y
                    break
            heights.append(col_height)
        for i in range(len(heights) - 1):
            bumpiness += abs(heights[i] - heights[i+1])
        return bumpiness

class TetrisLogic:
    """一人のテトリスプレイヤーのゲームロジックをカプセル化します。"""
    def __init__(self, opponent=None, line_clear_callback=None):
        self.opponent = opponent
        self.board = []
        self.current_piece = None
        self.next_piece = None
        self.held_piece = None
        self.can_hold = True
        self.score = 0
        self.game_over = False
        self.lines_cleared_total = 0
        self.bag = []
        self.combo_count = 0 
        self.garbage_queue = 0
        self.needs_board_redraw = True
        self.line_clear_callback = line_clear_callback
        self.reset()

    def reset(self):
        self.board = [['empty'] * BASE_BOARD_WIDTH for _ in range(BOARD_HEIGHT)]
        self.score = 0
        self.lines_cleared_total = 0
        self.game_over = False
        self.can_hold = True
        self.held_piece = None
        self.combo_count = 0
        self.garbage_queue = 0
        self.needs_board_redraw = True
        self.bag = list(STANDARD_TETROMINOS.keys()) * 2
        random.shuffle(self.bag)
        self.current_piece = self._get_new_piece()
        self.next_piece = self._get_new_piece()

    def _get_new_piece(self):
        if len(self.bag) < len(STANDARD_TETROMINOS.keys()):
            new_bag = list(STANDARD_TETROMINOS.keys())
            random.shuffle(new_bag)
            self.bag.extend(new_bag)
        shape_key = self.bag.pop(0)
        return {'shape': shape_key, 'rotation': 0, 'x': BASE_BOARD_WIDTH // 2 - 2, 'y': 0}

    def _is_valid_position(self, piece_shape, x_offset, y_offset):
        if not piece_shape: return False
        for y, row in enumerate(piece_shape):
            for x, cell in enumerate(row):
                if cell:
                    board_x, board_y = x + x_offset, y + y_offset
                    if not (0 <= board_x < BASE_BOARD_WIDTH and 0 <= board_y < BOARD_HEIGHT and self.board[board_y][board_x] == 'empty'):
                        return False
        return True

    def _get_ghost_y(self):
        ghost_y = self.current_piece['y']
        shape = self._get_current_shape_matrix()
        while self._is_valid_position(shape, self.current_piece['x'], ghost_y + 1):
            ghost_y += 1
        return ghost_y

    def _get_current_shape_matrix(self):
        if not self.current_piece: return []
        shape_key = self.current_piece['shape']
        rotations = STANDARD_TETROMINOS[shape_key]
        rot_index = self.current_piece['rotation']
        return rotations[rot_index % len(rotations)]

    def move(self, dx):
        if self.game_over or not self.current_piece: return
        shape = self._get_current_shape_matrix()
        if self._is_valid_position(shape, self.current_piece['x'] + dx, self.current_piece['y']):
            self.current_piece['x'] += dx

    def soft_drop(self):
        if self.game_over or not self.current_piece: return
        shape = self._get_current_shape_matrix()
        if self._is_valid_position(shape, self.current_piece['x'], self.current_piece['y'] + 1):
            self.current_piece['y'] += 1
            self.score += 1
        else:
            self._lock_piece()

    def rotate(self, clockwise=True):
        if self.game_over or not self.current_piece or self.current_piece['shape'] == 'O': return
        original_rotation = self.current_piece['rotation']
        self.current_piece['rotation'] += 1 if clockwise else -1
        shape = self._get_current_shape_matrix()
        if not self._is_valid_position(shape, self.current_piece['x'], self.current_piece['y']):
            if self._is_valid_position(shape, self.current_piece['x'] - 1, self.current_piece['y']):
                self.current_piece['x'] -= 1
            elif self._is_valid_position(shape, self.current_piece['x'] + 1, self.current_piece['y']):
                self.current_piece['x'] += 1
            else:
                self.current_piece['rotation'] = original_rotation

    def hard_drop(self):
        if self.game_over or not self.current_piece: return
        drop_y = self._get_ghost_y()
        self.score += (drop_y - self.current_piece['y']) * 2
        self.current_piece['y'] = drop_y
        self._lock_piece()

    def _lock_piece(self):
        shape = self._get_current_shape_matrix()
        for y, row in enumerate(shape):
            for x, cell in enumerate(row):
                if cell:
                    board_x = x + self.current_piece['x']
                    board_y = y + self.current_piece['y']
                    if 0 <= board_y < BOARD_HEIGHT:
                        self.board[board_y][board_x] = self.current_piece['shape']
        self.needs_board_redraw = True
        lines_cleared = self._clear_lines()
        if self.line_clear_callback and lines_cleared > 0:
            self.line_clear_callback(lines_cleared)
        if lines_cleared > 0:
            self.combo_count += 1
        else:
            self.combo_count = 0
        if self.opponent:
            cancelled_amount = min(self.garbage_queue, lines_cleared)
            self.garbage_queue -= cancelled_amount
            lines_cleared_after_cancel = lines_cleared - cancelled_amount
            self._send_attack(lines_cleared_after_cancel)
        if self.garbage_queue > 0 and lines_cleared == 0:
            self._apply_queued_garbage()
        self.current_piece = self.next_piece
        self.next_piece = self._get_new_piece()
        self.can_hold = True
        if not self._is_valid_position(self._get_current_shape_matrix(), self.current_piece['x'], self.current_piece['y']):
            self.game_over = True
            self.current_piece = None

    def _clear_lines(self):
        new_board = [row for row in self.board if any(cell == 'empty' for cell in row)]
        lines_cleared = BOARD_HEIGHT - len(new_board)
        if lines_cleared > 0:
            empty_rows = [['empty'] * BASE_BOARD_WIDTH for _ in range(lines_cleared)]
            self.board = empty_rows + new_board
            self.lines_cleared_total += lines_cleared
            score_map = {1: 100, 2: 300, 3: 500, 4: 800}
            self.score += score_map.get(lines_cleared, 0)
            if lines_cleared == 4:
                self.score += 400
        return lines_cleared

    def _send_attack(self, lines_cleared):
        if self.opponent is None or self.opponent.game_over:
            return
        attack_map = {1: 0, 2: 1, 3: 2, 4: 4}
        base_attack = attack_map.get(lines_cleared, 0)
        combo_attack = self.combo_count // 2
        garbage_to_send = base_attack + combo_attack
        if garbage_to_send > 0:
            self.opponent.add_garbage_lines(garbage_to_send)

    def add_garbage_lines(self, num_lines):
        self.garbage_queue += num_lines

    def _apply_queued_garbage(self):
        num_lines = self.garbage_queue
        hole_position = random.randint(0, BASE_BOARD_WIDTH - 1)
        for _ in range(num_lines):
            if len(self.board) >= BOARD_HEIGHT:
                self.board.pop(0) 
            garbage_row = ['garbage'] * BASE_BOARD_WIDTH
            garbage_row[hole_position] = 'empty'
            self.board.append(garbage_row)
        self.garbage_queue = 0
        self.needs_board_redraw = True

    def hold(self):
        if not self.can_hold or self.game_over or not self.current_piece: return
        self.can_hold = False
        if self.held_piece is None:
            self.held_piece = {'shape': self.current_piece['shape']}
            self.current_piece = self.next_piece
            self.next_piece = self._get_new_piece()
        else:
            held_shape = self.held_piece['shape']
            self.held_piece['shape'] = self.current_piece['shape']
            self.current_piece = {'shape': held_shape, 'rotation': 0, 'x': BASE_BOARD_WIDTH // 2 - 2, 'y': 0}
        if not self._is_valid_position(self._get_current_shape_matrix(), self.current_piece['x'], self.current_piece['y']):
                 self.game_over = True
                 self.current_piece = None

class TitleScreen(QWidget):
    """ゲームのメインメニュー画面。"""
    def __init__(self, app_instance, parent=None):
        super().__init__(parent)
        self.app_instance = app_instance
        self.init_ui()

    def paintEvent(self, event):
        painter = QPainter(self)
        pixmap = QPixmap(TITLE_BACKGROUND_IMAGE_PATH)
        if not pixmap.isNull():
            painter.drawPixmap(self.rect(), pixmap)
        else:
            painter.fillRect(self.rect(), QColor("#1a1a2e"))
        super().paintEvent(event)

    def init_ui(self):
        main_layout = QVBoxLayout(self)
        main_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        btn_style = """
            QPushButton {
                font-size: 32px; 
                padding: 20px 40px;
                color: white;
                border: 2px solid #00aaff;
                border-radius: 12px;
                font-weight: bold;
                min-width: 600px;
                background-color: rgba(2, 80, 145, 0.75);
            }
            QPushButton:hover {
                background-color: rgba(0, 119, 194, 0.85);
                border-color: #33ccff;
            }
            QPushButton:pressed {
                background-color: rgba(0, 56, 102, 0.9);
                padding-top: 22px; 
                padding-bottom: 18px;
            }
        """
        
        title_label = QLabel(self)
        if os.path.exists(TITLE_IMAGE_PATH):
            pixmap = QPixmap(TITLE_IMAGE_PATH)
            if pixmap.width() > 800:
                pixmap = pixmap.scaledToWidth(800, Qt.TransformationMode.SmoothTransformation)
            title_label.setPixmap(pixmap)
        else:
            print(f"[警告] タイトル画像が見つかりません: {TITLE_IMAGE_PATH}")
            title_label.setText("TetriaRival")
            title_label.setStyleSheet("font-size: 120px; font-weight: bold; color: #ffffff; text-shadow: 0 0 15px #00ffff, 0 0 25px #00ffff;")
        
        title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title_label.setStyleSheet("background: transparent;")

        button_layout = QVBoxLayout()
        button_layout.setSpacing(15) 
        button_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        btn_pvp = QPushButton(f"👥  {MODE_PVP}")
        btn_pvp.setStyleSheet(btn_style)
        btn_pvp.clicked.connect(lambda: self.handle_button_click(self.app_instance.show_game_screen, MODE_PVP))
        
        btn_pvc_normal = QPushButton("👤 vs 🤖 (ノーマル)")
        btn_pvc_normal.setStyleSheet(btn_style)
        btn_pvc_normal.clicked.connect(lambda: self.handle_button_click(self.app_instance.show_game_screen, MODE_PVC, speed='NORMAL'))

        btn_pvc_fast = QPushButton("👤 vs 🤖 (ファスト)")
        btn_pvc_fast.setStyleSheet(btn_style)
        btn_pvc_fast.clicked.connect(lambda: self.handle_button_click(self.app_instance.show_game_screen, MODE_PVC, speed='FAST'))

        btn_config = QPushButton("⚙️  操作設定")
        btn_config.setStyleSheet(btn_style)
        btn_config.clicked.connect(lambda: self.handle_button_click(self.app_instance.show_key_config_screen))

        btn_quit = QPushButton("⏻  終了")
        btn_quit.setStyleSheet(btn_style)
        btn_quit.clicked.connect(lambda: self.handle_button_click(QApplication.instance().quit))

        button_layout.addWidget(btn_pvp)
        button_layout.addWidget(btn_pvc_normal)
        button_layout.addWidget(btn_pvc_fast)
        button_layout.addWidget(btn_config)
        button_layout.addWidget(btn_quit)
        
        volume_container = QWidget()
        volume_container.setFixedWidth(400)
        volume_layout = QHBoxLayout(volume_container)
        volume_layout.setContentsMargins(10, 10, 10, 10)
        volume_layout.setSpacing(10)

        slider_style = """
            QSlider::groove:horizontal { border: 1px solid #bbb; background: white; height: 8px; border-radius: 4px; }
            QSlider::handle:horizontal { background: qlineargradient(x1:0, y1:0, x2:1, y2:1, stop:0 #b4b4b4, stop:1 #8f8f8f); border: 1px solid #5c5c5c; width: 18px; margin: -2px 0; border-radius: 9px; }
        """
        
        bgm_label = QLabel("BGM:")
        bgm_label.setStyleSheet("color: white; font-weight: bold;")
        self.bgm_slider = QSlider(Qt.Orientation.Horizontal)
        self.bgm_slider.setStyleSheet(slider_style)
        self.bgm_slider.setRange(0, 100)
        self.bgm_slider.setValue(int(self.app_instance.bgm_volume * 100))
        self.bgm_slider.valueChanged.connect(self.app_instance.set_bgm_volume)

        se_label = QLabel("SE:")
        se_label.setStyleSheet("color: white; font-weight: bold;")
        self.se_slider = QSlider(Qt.Orientation.Horizontal)
        self.se_slider.setStyleSheet(slider_style)
        self.se_slider.setRange(0, 100)
        self.se_slider.setValue(int(self.app_instance.se_volume * 100))
        self.se_slider.valueChanged.connect(self.app_instance.set_se_volume)

        volume_layout.addWidget(bgm_label)
        volume_layout.addWidget(self.bgm_slider)
        volume_layout.addSpacing(15)
        volume_layout.addWidget(se_label)
        volume_layout.addWidget(self.se_slider)
        
        bottom_left_layout = QHBoxLayout()
        bottom_left_layout.addWidget(volume_container)
        bottom_left_layout.addStretch()

        ### 変更点: スタッフロールボタンのフォントを太字に ###
        credits_button = QPushButton("StaffRoll")
        credits_button.setStyleSheet("""
            QPushButton { 
                font-size: 14px; 
                font-weight: bold;
                color: white; 
                background-color: rgba(0, 0, 0, 0.5); 
                padding: 5px 10px;
                border-radius: 5px;
                border: 1px solid white;
            }
            QPushButton:hover {
                background-color: rgba(255, 255, 255, 0.3);
            }
        """)
        credits_button.clicked.connect(self.app_instance.show_roll_credits_screen)
        
        bottom_right_layout = QHBoxLayout()
        bottom_right_layout.addStretch()
        bottom_right_layout.addWidget(credits_button)
        
        bottom_bar = QHBoxLayout()
        bottom_bar.addLayout(bottom_left_layout)
        bottom_bar.addLayout(bottom_right_layout)

        main_layout.addStretch(2)
        main_layout.addWidget(title_label)
        main_layout.addSpacing(40)
        main_layout.addLayout(button_layout)
        main_layout.addStretch(3)
        main_layout.addLayout(bottom_bar)

    def handle_button_click(self, action, *args, **kwargs):
        self.app_instance.play_decision_sound()
        action(*args, **kwargs)

class RollCreditsScreen(QWidget):
    def __init__(self, app_instance, parent=None):
        super().__init__(parent)
        self.app_instance = app_instance
        self.scroll_animation = QPropertyAnimation()
        self._background_pixmap = QPixmap(CREDITS_BACKGROUND_IMAGE_PATH)
        if self._background_pixmap.isNull():
            print(f"[警告] クレジット背景画像が見つかりません: {CREDITS_BACKGROUND_IMAGE_PATH}")
        self.init_ui()

    def paintEvent(self, event):
        painter = QPainter(self)
        if not self._background_pixmap.isNull():
            scaled_pixmap = self._background_pixmap.scaled(self.size(), Qt.AspectRatioMode.KeepAspectRatioByExpanding, Qt.TransformationMode.SmoothTransformation)
            x = (self.width() - scaled_pixmap.width()) // 2
            y = (self.height() - scaled_pixmap.height()) // 2
            painter.drawPixmap(x, y, scaled_pixmap)
        else:
            painter.fillRect(self.rect(), Qt.GlobalColor.black)
        super().paintEvent(event)

    def init_ui(self):
        self.setStyleSheet("background-color: transparent;")
        self.scroll_container = QWidget(self)
        self.scroll_container.setStyleSheet("background-color: transparent;")
        container_layout = QVBoxLayout(self.scroll_container)
        container_layout.setContentsMargins(20, 20, 20, 20)
        container_layout.setSpacing(20)
        container_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)

        ### 変更点: 先頭の画像をタイトルロゴに変更 ###
        logo_label = QLabel()
        if os.path.exists(TITLE_IMAGE_PATH):
            pixmap = QPixmap(TITLE_IMAGE_PATH)
            # クレジット用に少し小さめにスケーリング
            if pixmap.width() > 600:
                 pixmap = pixmap.scaledToWidth(600, Qt.TransformationMode.SmoothTransformation)
            logo_label.setPixmap(pixmap)
        logo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        container_layout.addWidget(logo_label)

        credits_label = QLabel()
        # HTML内の画像パスもresource_pathで解決し、CSSが使えるように/に置換
        app_icon_html = resource_path(APP_ICON_PATH).replace(os.sep, '/')
        gcs2_logo_html = resource_path(GCS2_LOGO_PATH).replace(os.sep, '/')
        sapocreate_logo_html = resource_path(SAPOCREATE_LOGO_PATH).replace(os.sep, '/')
        gcot_logo_html = resource_path(GCOT_LOGO_PATH).replace(os.sep, '/')
        
        credits_text = f"""
        <br>
        <h1 align="center"><font color="white">TetriaRival</font></h1>
        <br>
        <p align="center"><img src="{app_icon_html}" width="120"></p>
        <br>
        <h2 align="center"><font color="white">開発者</font></h2>
        <p align="center"><font color="white" size="+1">SECA Project</font></p>
        <p align="center"><font color="white">GCS-97 サポ / from Kagawa</font></p>
        <p align="center"><font color="white">GCS-97 initial members (joined in 2023)</font></p>
        <p align="center">
          <span style="background-color: #005588; padding: 3px 8px; border-radius: 5px;">
            <font color="white">Sub-admin of GCS-97</font>
          </span>
        </p>
        <br><br><br><br>
        <h2 align="center"><font color="white">Special Thanks</font></h2>
        <p align="center"><font color="white">Google Gemini</font></p>
        <p align="center"><font color="white">PyQt Project</font></p>
        <p align="center"><font color="white">Pygame Project</font></p>
        <br><br><br><br>
        <h2 align="center"><font color="white">使用音源</font></h2>
        <p align="center"><font color="white">待機画面BGM: "あふれるアイデア"</font></p>
        <p align="center"><font color="white">プレイ画面BGM: "不合理スクランブル"</font></p>
        <p align="center"><font color="white">スタッフロールBGM: "眠れる世界樹"</font></p>
        <p align="center"><font color="white">制作: BGMer</font></p>
        <p align="center"><font color="white">URL: https://bgmer.net</font></p>
        <br><br>
        <p align="center"><font color="white">選択画面SE: "カーソル移動2", "決定ボタンを押す7", etc.</font></p>
        <p align="center"><font color="white">プレイ中SE: "カーソル移動5", "電子ルーレット停止ボタンを押す", etc.</font></p>
        <p align="center"><font color="white">制作: 効果音ラボ</font></p>
        <p align="center"><font color="white">URL: https://soundeffect-lab.info</font></p>
        <br><br><br><br>
        <h2 align="center"><font color="white">開発チーム</font></h2>
        <p align="center"><img src="{gcs2_logo_html}" width="200"></p>
        <h3 align="center"><font color="white">GCS-97(Global Create Station)</font></p>
        <h3 align="center"><font color="white">Established in 2023</font></p>
        <br><br>
        <p align="center"><img src="{sapocreate_logo_html}" width="200"></p>
        <h3 align="center"><font color="white">サポクリエイト</font></p>
        <br><br>
        <p align="center"><img src="{gcot_logo_html}" width="200"></p>
        <h3 align="center"><font color="white">GCOT supported by GCS-97</font></p>
        <br><br><br><br>
        <h3 align="center"><font color="white">Copyright © 2025 GCS-97/(SAPOCreate,GCOT). <br>
        All Rights Reserved.</font></h3>
        <br><br><br><br><br><br><br><br>
        """
        credits_label.setText(credits_text)
        credits_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        credits_label.setStyleSheet("background-color: transparent; font-size: 18px;")
        credits_label.setTextFormat(Qt.TextFormat.RichText)
        credits_label.setWordWrap(True)
        container_layout.addWidget(credits_label)

        self.mousePressEvent = self.skip_credits

    def showEvent(self, event):
        super().showEvent(event)
        QTimer.singleShot(10, self.start_credits_animation)

    def hideEvent(self, event):
        super().hideEvent(event)
        self.stop_credits_animation()

    def start_credits_animation(self):
        self.stop_credits_animation()
        ### 変更点: コンテナの幅を親ウィジェットに合わせる ###
        self.scroll_container.setFixedWidth(self.width())
        self.scroll_container.adjustSize()
        container_height = self.scroll_container.height()
        start_x = (self.width() - self.scroll_container.width()) // 2
        start_y = self.height()
        end_y = -container_height
        self.scroll_container.move(start_x, start_y)
        self.scroll_animation = QPropertyAnimation(self.scroll_container, b"pos")
        self.scroll_animation.setDuration(70000)
        self.scroll_animation.setStartValue(QPoint(start_x, start_y))
        self.scroll_animation.setEndValue(QPoint(start_x, end_y))
        self.scroll_animation.finished.connect(self.app_instance.show_title_screen)
        self.scroll_animation.start()

    def stop_credits_animation(self):
        if self.scroll_animation and self.scroll_animation.state() == QPropertyAnimation.State.Running:
            self.scroll_animation.stop()

    def skip_credits(self, event):
        self.stop_credits_animation()
        self.app_instance.show_title_screen()

    def resizeEvent(self, event):
        super().resizeEvent(event)
        if self.isVisible():
            QTimer.singleShot(10, self.start_credits_animation)

class GameScreen(QWidget):
    """テトリスをプレイするメインのゲーム画面。"""
    def __init__(self, key_config, app_instance, parent=None):
        super().__init__(parent)
        self.app_instance = app_instance
        self.key_config = key_config
        
        self.p1_logic = TetrisLogic(line_clear_callback=self.app_instance.play_line_clear_se)
        self.p2_logic = TetrisLogic(line_clear_callback=self.app_instance.play_line_clear_se)
        
        self.p1_logic.opponent = self.p2_logic
        self.p2_logic.opponent = self.p1_logic
        self.cpu = None 
        self.cpu_target_move = None 
        self.cpu_last_piece = None 
        self.game_timer = QTimer(self)
        self.game_timer.timeout.connect(self.update_game)
        self.init_ui()
        
    def init_ui(self):
        main_layout = QHBoxLayout(self)
        main_layout.setSpacing(20)
        main_layout.setContentsMargins(20, 20, 20, 20)
        main_layout.setAlignment(Qt.AlignmentFlag.AlignTop)
        self.setStyleSheet("background-color: #1a1a2e;")
        
        self.p1_elements = self.create_player_area("プレイヤー 1")
        p1_container = QWidget()
        p1_container.setLayout(self.p1_elements['layout'])
        main_layout.addWidget(p1_container)
        
        separator = QFrame()
        separator.setFrameShape(QFrame.Shape.VLine)
        separator.setFrameShadow(QFrame.Shadow.Sunken)
        separator.setStyleSheet("background-color: #555599;")
        main_layout.addWidget(separator)

        self.p2_elements = self.create_player_area("プレイヤー 2")
        p2_container = QWidget()
        p2_container.setLayout(self.p2_elements['layout'])
        main_layout.addWidget(p2_container)


    def create_player_area(self, player_name):
        player_main_layout = QVBoxLayout()
        player_main_layout.setSpacing(10)

        top_info_layout = QHBoxLayout()
        name_label = QLabel(player_name)
        name_label.setStyleSheet("font-size: 24px; font-weight: bold; color: white;")
        score_label = QLabel("スコア: 0")
        score_label.setStyleSheet("font-size: 18px; color: white;")
        top_info_layout.addWidget(name_label)
        top_info_layout.addStretch()
        top_info_layout.addWidget(score_label)
        
        combo_label = QLabel("")
        combo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        combo_label.setStyleSheet("font-size: 36px; font-weight: bold; color: #FFA500;")
        combo_label.setVisible(False)

        game_area_layout = QHBoxLayout()
        
        hold_area = self.create_preview_area("ホールド")
        
        game_scene = QGraphicsScene(0, 0, BASE_BOARD_WIDTH * BLOCK_SIZE, BOARD_HEIGHT * BLOCK_SIZE)
        game_view = QGraphicsView(game_scene)
        game_view.setFixedSize(BASE_BOARD_WIDTH * BLOCK_SIZE + 4, BOARD_HEIGHT * BLOCK_SIZE + 4)
        game_view.setStyleSheet("background-color: black; border: 2px solid #555599;")
        game_view.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        game_view.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)

        right_sidebar_layout = QVBoxLayout()
        next_area = self.create_preview_area("ネクスト")
        garbage_label = QLabel("")
        garbage_label.setStyleSheet("font-size: 48px; font-weight: bold; color: red;")
        garbage_label.setAlignment(Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignTop)
        garbage_label.setVisible(False)
        
        right_sidebar_layout.addWidget(next_area['box'])
        right_sidebar_layout.addSpacing(20)
        right_sidebar_layout.addWidget(garbage_label)
        right_sidebar_layout.addStretch()

        game_area_layout.addWidget(hold_area['box'], alignment=Qt.AlignmentFlag.AlignTop)
        game_area_layout.addWidget(game_view)
        game_area_layout.addLayout(right_sidebar_layout)

        result_label = QLabel("")
        result_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        result_label.setVisible(False)
        
        player_main_layout.addLayout(top_info_layout)
        player_main_layout.addWidget(combo_label)
        player_main_layout.addLayout(game_area_layout)
        player_main_layout.addWidget(result_label)

        return {
            'layout': player_main_layout, 'name': name_label, 'score': score_label, 
            'scene': game_scene, 'view': game_view,
            'hold_scene': hold_area['scene'], 'next_scene': next_area['scene'],
            'result_label': result_label,
            'combo_label': combo_label,
            'garbage_label': garbage_label,
            'dynamic_items': []
        }

    def create_preview_area(self, label_text):
        box = QWidget()
        layout = QVBoxLayout(box)
        layout.setContentsMargins(0,0,0,0)
        label = QLabel(label_text)
        label.setStyleSheet("font-size: 18px; color: white; qproperty-alignment: 'AlignCenter';")
        
        scene = QGraphicsScene(0, 0, 4 * BLOCK_SIZE, 4 * BLOCK_SIZE)
        view = QGraphicsView(scene)
        view.setFixedSize(4 * BLOCK_SIZE + 2, 4 * BLOCK_SIZE + 2)
        view.setStyleSheet("background-color: #111; border: 1px solid #555599;")
        view.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        view.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        
        layout.addWidget(label)
        layout.addWidget(view)
        return {'box': box, 'scene': scene}

    def start_game(self, mode, speed='NORMAL'):
        print(f"--- ゲーム開始: {mode}, Speed: {speed} ---")
        self.p1_logic.reset()
        self.p2_logic.reset()
        self.p1_elements['result_label'].setVisible(False)
        self.p2_elements['result_label'].setVisible(False)
        self.p1_elements['combo_label'].setVisible(False)
        self.p2_elements['combo_label'].setVisible(False)
        self.p1_elements['garbage_label'].setVisible(False)
        self.p2_elements['garbage_label'].setVisible(False)
        
        self.cpu = None
        self.cpu_speed = None
        self.cpu_target_move = None
        self.cpu_last_piece = None

        if mode == MODE_PVC:
            self.p2_elements['name'].setText("CPU")
            self.cpu = TetrisAI(self.p2_logic)
            self.cpu_speed = speed 
        else:
            self.p2_elements['name'].setText("プレイヤー 2")

        self.game_timer.start(500)
        self.setFocus()
        self.update_ui()

    def update_game(self):
        if self.game_timer.isActive():
            if not self.p1_logic.game_over: self.p1_logic.soft_drop()
            
            if self.cpu and not self.p2_logic.game_over:
                if self.p2_logic.current_piece is not self.cpu_last_piece:
                    self.cpu_last_piece = self.p2_logic.current_piece
                    self.cpu_target_move = self.cpu.find_best_move()
                
                if self.cpu_target_move:
                    if self.cpu_speed == 'FAST':
                        if self.cpu_target_move['hold'] and self.p2_logic.can_hold:
                            self.p2_logic.hold()
                            self.cpu_target_move = self.cpu.find_best_move() 
                            if not self.cpu_target_move: return
                        
                        if self.p2_logic.current_piece:
                            self.p2_logic.current_piece['rotation'] = self.cpu_target_move['rotation']
                            self.p2_logic.current_piece['x'] = self.cpu_target_move['x']
                            self.p2_logic.hard_drop()
                    
                    elif self.cpu_speed == 'NORMAL':
                        if self.cpu_target_move['hold'] and self.p2_logic.can_hold:
                            self.p2_logic.hold()
                            self.cpu_target_move = None
                        elif self.p2_logic.current_piece:
                            target_rot = self.cpu_target_move['rotation']
                            target_x = self.cpu_target_move['x']
                            current_rot = self.p2_logic.current_piece['rotation']
                            current_x = self.p2_logic.current_piece['x']

                            if len(STANDARD_TETROMINOS[self.p2_logic.current_piece['shape']]) > 1 and current_rot % 4 != target_rot % 4:
                                self.p2_logic.current_piece['rotation'] = target_rot
                            
                            if current_x < target_x:
                                self.p2_logic.move(1)
                            elif current_x > target_x:
                                self.p2_logic.move(-1)

                            self.p2_logic.soft_drop()

            elif not self.p2_logic.game_over:
                self.p2_logic.soft_drop()

            self.update_ui()
            
            p1_dead = self.p1_logic.game_over
            p2_dead = self.p2_logic.game_over

            if p1_dead and p2_dead: self.end_game('draw')
            elif p1_dead: self.end_game('p2')
            elif p2_dead: self.end_game('p1')

    def end_game(self, winner):
        self.game_timer.stop()
        win_style = "font-size: 80px; font-weight: bold; color: #FFD700;"
        lose_style = "font-size: 80px; font-weight: bold; color: #C0C0C0;"
        draw_style = "font-size: 80px; font-weight: bold; color: #FFFFFF;"
        winner_name = "プレイヤー1" if winner == 'p1' else "CPU" if self.cpu else "プレイヤー2"
        if winner == 'p1':
            self.p1_elements['result_label'].setText("WIN")
            self.p1_elements['result_label'].setStyleSheet(win_style)
            self.p2_elements['result_label'].setText("LOSE")
            self.p2_elements['result_label'].setStyleSheet(lose_style)
        elif winner == 'p2':
            self.p2_elements['result_label'].setText("WIN")
            self.p2_elements['result_label'].setStyleSheet(win_style)
            self.p1_elements['result_label'].setText("LOSE")
            self.p1_elements['result_label'].setStyleSheet(lose_style)
        elif winner == 'draw':
            self.p1_elements['result_label'].setText("DRAW")
            self.p1_elements['result_label'].setStyleSheet(draw_style)
            self.p2_elements['result_label'].setText("DRAW")
            self.p2_elements['result_label'].setStyleSheet(draw_style)
        self.p1_elements['result_label'].setVisible(True)
        self.p2_elements['result_label'].setVisible(True)

    def update_ui(self):
        self.draw_player_state(self.p1_logic, self.p1_elements)
        self.draw_player_state(self.p2_logic, self.p2_elements)

    def draw_player_state(self, logic, elements):
        for item in elements['dynamic_items']:
            elements['scene'].removeItem(item)
        elements['dynamic_items'].clear()
        elements['score'].setText(f"スコア: {logic.score}")
        if logic.needs_board_redraw:
            self.draw_board(logic, elements['scene'])
            logic.needs_board_redraw = False
        if not logic.game_over and logic.current_piece:
            self.draw_piece(logic, elements['scene'], elements['dynamic_items'])
            self.draw_ghost_piece(logic, elements['scene'], elements['dynamic_items'])
        elements['next_scene'].clear()
        if logic.next_piece:
            self.draw_preview_piece(logic.next_piece, elements['next_scene'])
        elements['hold_scene'].clear()
        if logic.held_piece:
            self.draw_preview_piece(logic.held_piece, elements['hold_scene'])
        if logic.combo_count > 1:
            elements['combo_label'].setText(f"{logic.combo_count} COMBO")
            elements['combo_label'].setVisible(True)
        else:
            elements['combo_label'].setVisible(False)
        if logic.garbage_queue > 0:
            elements['garbage_label'].setText(str(logic.garbage_queue))
            elements['garbage_label'].setVisible(True)
        else:
            elements['garbage_label'].setVisible(False)
            
    def draw_board(self, logic, scene):
        items_to_remove = [item for item in scene.items() if item.data(0) == 'static']
        for item in items_to_remove:
            scene.removeItem(item)
        for y, row in enumerate(logic.board):
            for x, cell in enumerate(row):
                color = TETROMINO_COLORS[cell] if cell != 'empty' else TETROMINO_COLORS['empty']
                rect = scene.addRect(x * BLOCK_SIZE, y * BLOCK_SIZE, BLOCK_SIZE, BLOCK_SIZE, QPen(Qt.PenStyle.NoPen), QBrush(color))
                rect.setData(0, 'static')

    def draw_piece(self, logic, scene, dynamic_items):
        shape_matrix = logic._get_current_shape_matrix()
        color = TETROMINO_COLORS[logic.current_piece['shape']]
        for y, row in enumerate(shape_matrix):
            for x, cell in enumerate(row):
                if cell:
                    px = (logic.current_piece['x'] + x) * BLOCK_SIZE
                    py = (logic.current_piece['y'] + y) * BLOCK_SIZE
                    rect = scene.addRect(px, py, BLOCK_SIZE, BLOCK_SIZE, QPen(QColor("white"), 0.5), QBrush(color))
                    rect.setData(0, 'dynamic')
                    dynamic_items.append(rect) 
                    
    def draw_ghost_piece(self, logic, scene, dynamic_items):
        ghost_y = logic._get_ghost_y()
        shape_matrix = logic._get_current_shape_matrix()
        color = TETROMINO_COLORS['ghost']
        for y, row in enumerate(shape_matrix):
            for x, cell in enumerate(row):
                if cell:
                    px = (logic.current_piece['x'] + x) * BLOCK_SIZE
                    py = (ghost_y + y) * BLOCK_SIZE
                    rect = scene.addRect(px, py, BLOCK_SIZE, BLOCK_SIZE, QPen(Qt.PenStyle.NoPen), QBrush(color))
                    rect.setData(0, 'dynamic')
                    dynamic_items.append(rect)

    def draw_preview_piece(self, piece, scene):
        shape_key = piece['shape']
        shape_matrix = STANDARD_TETROMINOS[shape_key][0]
        color = TETROMINO_COLORS[shape_key]
        offset_x = (4 - len(shape_matrix[0])) / 2.0
        offset_y = (4 - len(shape_matrix)) / 2.0
        for y, row in enumerate(shape_matrix):
            for x, cell in enumerate(row):
                if cell:
                    px = (x + offset_x) * BLOCK_SIZE
                    py = (y + offset_y) * BLOCK_SIZE
                    scene.addRect(px, py, BLOCK_SIZE, BLOCK_SIZE, QPen(QColor("white"), 0.5), QBrush(color))

    def keyPressEvent(self, event):
        if self.game_timer.isActive():
            key = event.key()
            p1_key_set = 'p1_cpu' if self.cpu else 'p1'
            p1_keys = self.key_config[p1_key_set]
            if key == p1_keys['left']: self.p1_logic.move(-1)
            elif key == p1_keys['right']: self.p1_logic.move(1)
            elif key == p1_keys['down']: self.p1_logic.soft_drop()
            elif key == p1_keys['rotate_right']: self.p1_logic.rotate(True)
            elif key == p1_keys['rotate_left']: self.p1_logic.rotate(False)
            elif key == p1_keys['hard_drop']: self.p1_logic.hard_drop()
            elif key == p1_keys['hold']: self.p1_logic.hold()
            
            if not self.cpu:
                p2_keys = self.key_config['p2']
                if key == p2_keys['left']: self.p2_logic.move(-1)
                elif key == p2_keys['right']: self.p2_logic.move(1)
                elif key == p2_keys['down']: self.p2_logic.soft_drop()
                elif key == p2_keys['rotate_right']: self.p2_logic.rotate(True)
                elif key == p2_keys['rotate_left']: self.p2_logic.rotate(False)
                elif key == p2_keys['hard_drop']: self.p2_logic.hard_drop()
                elif key == p2_keys['hold']: self.p2_logic.hold()

            self.update_ui()
        
        if event.key() == Qt.Key.Key_Escape:
            self.game_timer.stop()
            self.app_instance.play_decision_sound()
            self.app_instance.show_title_screen()

class KeyConfigScreen(QWidget):
    """プレイヤーのキーバインドを設定するための画面。"""
    def __init__(self, app_instance, parent=None):
        super().__init__(parent)
        self.app_instance = app_instance
        self.key_labels = {}
        self.capturing_key_for = None
        self.init_ui()

    def init_ui(self):
        main_layout = QVBoxLayout(self)
        main_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.setStyleSheet("background-color: #f0f0f0;")

        title_label = QLabel("操作設定")
        title_label.setStyleSheet("font-size: 48px; font-weight: bold; color: #333; margin-bottom: 30px;")
        main_layout.addWidget(title_label, alignment=Qt.AlignmentFlag.AlignCenter)

        config_layout = QHBoxLayout()
        config_layout.setSpacing(40)
        config_layout.addWidget(self.create_player_config_ui('p1', "対人戦 (1P)"))
        config_layout.addWidget(self.create_player_config_ui('p2', "対人戦 (2P)"))
        config_layout.addWidget(self.create_player_config_ui('p1_cpu', "CPU対戦 (1P)"))
        main_layout.addLayout(config_layout)
        main_layout.addStretch(1)
        
        btn_save_style = """
            QPushButton {
                font-size: 24px; padding: 12px 24px; color: white; font-weight: bold;
                background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #007bff, stop:1 #0056b3);
                border: 1px solid #0056b3; border-radius: 8px;
            }
            QPushButton:hover {
                background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #008cff, stop:1 #0060c3);
                border: 1px solid #0060c3;
            }
            QPushButton:pressed { background-color: #0056b3; }
        """
        btn_back_style = """
            QPushButton {
                font-size: 24px; padding: 12px 24px; color: white; font-weight: bold;
                background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #6c757d, stop:1 #5a6268);
                border: 1px solid #5a6268; border-radius: 8px;
            }
            QPushButton:hover {
                background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #7c858d, stop:1 #6a7278);
                border: 1px solid #6a7278;
            }
            QPushButton:pressed { background-color: #5a6268; }
        """
        btn_save = QPushButton("💾 設定を保存")
        btn_save.setStyleSheet(btn_save_style)
        btn_save.clicked.connect(self.save_config_and_return)
        
        btn_back = QPushButton("⏪ タイトルに戻る")
        btn_back.setStyleSheet(btn_back_style)
        btn_back.clicked.connect(self.back_to_title)
        
        btn_layout = QHBoxLayout()
        btn_layout.addStretch(1)
        btn_layout.addWidget(btn_save)
        btn_layout.addSpacing(20)
        btn_layout.addWidget(btn_back)
        btn_layout.addStretch(1)
        main_layout.addLayout(btn_layout)

    def create_player_config_ui(self, player_key, display_name):
        container = QFrame()
        container.setFrameShape(QFrame.Shape.StyledPanel)
        container.setStyleSheet("background-color: white; border-radius: 10px; border: 1px solid #ccc;")
        container.setMinimumWidth(360) 
        layout = QVBoxLayout(container)
        layout.setContentsMargins(20, 20, 20, 20)
        
        player_label = QLabel(display_name)
        player_label.setStyleSheet("font-size: 28px; font-weight: bold; color: #333; margin-bottom: 15px; border: none;")
        layout.addWidget(player_label, alignment=Qt.AlignmentFlag.AlignCenter)

        key_actions = {
            'left': "左移動", 'right': "右移動", 'down': "ソフトドロップ",
            'rotate_right': "右回転", 'rotate_left': "左回転",
            'hard_drop': "ハードドロップ", 'hold': "ホールド",
        }
        self.key_labels[player_key] = {}

        for action, name in key_actions.items():
            h_layout = QHBoxLayout()
            action_label = QLabel(f"{name}:")
            action_label.setStyleSheet("font-size: 20px; color: #333;")
            
            key_display_label = QLabel("...")
            key_display_label.setStyleSheet("border: 1px solid #ccc; padding: 8px; min-width: 140px; background-color: #f8f9fa; color: #333; font-size: 20px; font-weight: bold; border-radius: 5px;")
            key_display_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            
            key_display_label.mousePressEvent = lambda event, p=player_key, a=action: self.start_key_capture(p, a)
            
            self.key_labels[player_key][action] = key_display_label
            h_layout.addWidget(action_label)
            h_layout.addStretch(1)
            h_layout.addWidget(key_display_label)
            layout.addLayout(h_layout)
        return container

    def update_display(self):
        current_keys = self.app_instance.key_config
        for player_key in ['p1', 'p2', 'p1_cpu']:
            if player_key not in self.key_labels: continue
            for action, label in self.key_labels[player_key].items():
                key_code = current_keys.get(player_key, {}).get(action, 0)
                label.setText(key_to_string(key_code))
                label.setStyleSheet("border: 1px solid #ccc; padding: 8px; min-width: 140px; background-color: #f8f9fa; color: #333; font-size: 20px; font-weight: bold; border-radius: 5px;")
        
        self.capturing_key_for = None
        self.setFocus()
        
    def start_key_capture(self, player_key, action):
        self.app_instance.play_decision_sound()
        if self.capturing_key_for:
            p, a = self.capturing_key_for
            self.key_labels[p][a].setText(key_to_string(self.app_instance.key_config[p][a]))
            self.key_labels[p][a].setStyleSheet("border: 1px solid #ccc; padding: 8px; min-width: 140px; background-color: #f8f9fa; color: #333; font-size: 20px; font-weight: bold; border-radius: 5px;")

        self.capturing_key_for = (player_key, action)
        self.key_labels[player_key][action].setText("入力待機...")
        self.key_labels[player_key][action].setStyleSheet("border: 2px solid #d32f2f; padding: 8px; min-width: 140px; background-color: #ffcdd2; color: #d32f2f; font-size: 20px; font-weight: bold; border-radius: 5px;")

    def keyPressEvent(self, event):
        if self.capturing_key_for:
            player_key, action = self.capturing_key_for
            key_code = event.key()
            if key_code != Qt.Key.Key_Escape:
                self.app_instance.key_config[player_key][action] = key_code
            self.update_display()
            event.accept()
            self.app_instance.play_decision_sound()
        else:
            super().keyPressEvent(event)

    def save_config_and_return(self):
        self.app_instance.play_decision_sound()
        self.app_instance.save_key_config()
        QMessageBox.information(self, "キー設定", "キー設定を保存しました。")
        self.app_instance.show_title_screen()
    
    def back_to_title(self):
        self.app_instance.play_decision_sound()
        self.app_instance.show_title_screen()


class TetrisApp(QWidget):
    """画面とグローバルな状態を管理するメインアプリケーションクラス。"""
    def __init__(self):
        super().__init__()
        self.key_config = {}
        self.bgm_volume = 0.5
        self.se_volume = 1.0
        self.current_music_path = None
        
        self.line_clear_1_3_sound = None
        self.tetris_sound = None
        self.cursor_sound = None
        self.decision_sound = None
        
        self.load_volume_config()
        self.init_pygame()
        self.load_key_config()
        self.init_ui()

    def init_pygame(self):
        try:
            pygame.mixer.init()
            pygame.mixer.set_num_channels(32)
            print("[DEBUG] Pygameミキサーが初期化され、チャンネル数が32に設定されました。")
            
            self.line_clear_1_3_sound = self.load_sound(LINE_CLEAR_1_3_SE_PATH)
            self.tetris_sound = self.load_sound(TETRIS_SE_PATH)
            self.cursor_sound = self.load_sound(CURSOR_MOVE_SE_PATH)
            self.decision_sound = self.load_sound(DECISION_SE_PATH)
            
            self.set_bgm_volume(self.bgm_volume * 100)
            self.set_se_volume(self.se_volume * 100)
                
        except pygame.error as e:
            print(f"[エラー] PygameミキサーまたはSEの初期化エラー: {e}")
            
    def load_sound(self, path):
        absolute_path = os.path.abspath(path)
        print(f"[DEBUG] SEファイルを探しています: {absolute_path}")
        if not os.path.exists(path):
            print(f"[警告] SEファイルが見つかりません: {path}")
            return None
        try:
            sound = pygame.mixer.Sound(path)
            return sound
        except pygame.error as e:
            print(f"[エラー] SEロード失敗: {path} - {e}")
            return None

    def init_ui(self):
        self.setWindowTitle("TetriaRival")
        self.setFixedSize(1200, 750)
        if os.path.exists(APP_ICON_PATH):
            self.setWindowIcon(QIcon(APP_ICON_PATH))

        self.stacked_widget = QStackedWidget(self)
        self.title_screen = TitleScreen(self)
        self.game_screen = GameScreen(self.key_config, self)
        self.key_config_screen = KeyConfigScreen(self)
        self.roll_credits_screen = RollCreditsScreen(self) # スタッフロール画面を追加
        
        self.stacked_widget.addWidget(self.title_screen)
        self.stacked_widget.addWidget(self.game_screen)
        self.stacked_widget.addWidget(self.key_config_screen)
        self.stacked_widget.addWidget(self.roll_credits_screen)

        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.addWidget(self.stacked_widget)
        self.setLayout(main_layout)

        self.show_title_screen()
        QApplication.instance().aboutToQuit.connect(self.save_volume_config)
        QApplication.instance().aboutToQuit.connect(pygame.mixer.quit)

    def set_bgm_volume(self, value):
        volume = value / 100.0
        self.bgm_volume = volume
        pygame.mixer.music.set_volume(self.bgm_volume)

    def set_se_volume(self, value):
        volume = value / 100.0
        self.se_volume = volume
        for sound in [self.line_clear_1_3_sound, self.tetris_sound, self.cursor_sound, self.decision_sound]:
            if sound:
                sound.set_volume(self.se_volume)

    def save_volume_config(self):
        try:
            with open(VOLUME_CONFIG_FILE, 'w') as f:
                json.dump({'bgm_volume': self.bgm_volume, 'se_volume': self.se_volume}, f)
            print("[DEBUG] 音量設定を保存しました。")
        except Exception as e:
            print(f"[エラー] 音量設定の保存に失敗しました: {e}")

    def load_volume_config(self):
        if os.path.exists(VOLUME_CONFIG_FILE):
            try:
                with open(VOLUME_CONFIG_FILE, 'r') as f:
                    config = json.load(f)
                    self.bgm_volume = config.get('bgm_volume', 0.5)
                    self.se_volume = config.get('se_volume', 1.0)
                print("[DEBUG] 音量設定を読み込みました。")
            except Exception as e:
                print(f"[エラー] 音量設定の読み込みに失敗しました: {e}")
        else:
            print("[情報] 音量設定ファイルが見つからないため、デフォルト値を使用します。")

    def play_line_clear_se(self, lines_cleared):
        if lines_cleared == 4 and self.tetris_sound:
            self.tetris_sound.play()
        elif 0 < lines_cleared < 4 and self.line_clear_1_3_sound:
            self.line_clear_1_3_sound.play()
            
    def play_cursor_sound(self):
        if self.cursor_sound:
            self.cursor_sound.play()

    def play_decision_sound(self):
        if self.decision_sound:
            self.decision_sound.play()

    def show_title_screen(self):
        self.stacked_widget.setCurrentWidget(self.title_screen)
        self.play_music(TITLE_BGM_PATH, "タイトルBGM")

    def show_game_screen(self, mode, speed='NORMAL'):
        self.stacked_widget.setCurrentWidget(self.game_screen)
        self.game_screen.start_game(mode, speed)
        self.play_music(GAME_BGM_PATH, "ゲームBGM")

    def show_key_config_screen(self):
        self.key_config_screen.update_display()
        self.stacked_widget.setCurrentWidget(self.key_config_screen)
        # BGMはタイトル画面のものを継続
        self.play_music(TITLE_BGM_PATH, "タイトルBGM")

    def show_roll_credits_screen(self):
        self.stacked_widget.setCurrentWidget(self.roll_credits_screen)
        self.play_music(CREDITS_BGM_PATH, "クレジットBGM")

    def play_music(self, music_path, music_name="Music"):
        if self.current_music_path == music_path and pygame.mixer.music.get_busy():
            print(f"[DEBUG] {music_name} は既に再生中です。")
            return

        absolute_path = os.path.abspath(music_path)
        print(f"[DEBUG] BGMファイルを探しています: {absolute_path}")
        if not os.path.exists(music_path):
            print(f"[警告] {music_name} を再生できません: ファイルが見つかりません {music_path}")
            pygame.mixer.music.stop()
            self.current_music_path = None
            return
        try:
            pygame.mixer.music.load(music_path)
            pygame.mixer.music.set_volume(self.bgm_volume)
            pygame.mixer.music.play(-1)
            self.current_music_path = music_path
            print(f"[DEBUG] {music_name} を再生中。")
        except pygame.error as e:
            print(f"[エラー] Pygame {music_name} の読み込み/再生エラー: {e}")

    def load_key_config(self):
        self.key_config = {
            'p1': DEFAULT_KEY_CONFIG['p1'].copy(),
            'p2': DEFAULT_KEY_CONFIG['p2'].copy(),
            'p1_cpu': DEFAULT_KEY_CONFIG['p1_cpu'].copy()
        }
        if os.path.exists(KEY_CONFIG_FILE_RIVAL):
            try:
                with open(KEY_CONFIG_FILE_RIVAL, 'r', encoding='utf-8') as f:
                    player_key = None
                    for line in f:
                        line = line.strip()
                        if line == '[p1]': player_key = 'p1'
                        elif line == '[p2]': player_key = 'p2'
                        elif line == '[p1_cpu]': player_key = 'p1_cpu'
                        elif player_key and '=' in line:
                            action, key_code_str = line.split('=', 1)
                            if player_key in self.key_config and action.strip() in self.key_config[player_key]:
                                self.key_config[player_key][action.strip()] = int(key_code_str.strip())
            except Exception as e:
                print(f"[エラー] キー設定ファイルの読み込みエラー: {e}")

    def save_key_config(self):
        os.makedirs(os.path.dirname(KEY_CONFIG_FILE_RIVAL), exist_ok=True)
        try:
            with open(KEY_CONFIG_FILE_RIVAL, 'w', encoding='utf-8') as f:
                f.write('[p1]\n')
                for action, key in self.key_config['p1'].items():
                    f.write(f'{action} = {key}\n')
                f.write('\n[p2]\n')
                for action, key in self.key_config['p2'].items():
                    f.write(f'{action} = {key}\n')
                f.write('\n[p1_cpu]\n')
                for action, key in self.key_config['p1_cpu'].items():
                    f.write(f'{action} = {key}\n')
            print("[DEBUG] キー設定が保存されました。")
        except Exception as e:
            print(f"[エラー] キー設定ファイルの保存エラー: {e}")


if __name__ == "__main__":
    # エラーを防ぐためにアセットディレクトリが存在することを確認
    os.makedirs("tect/TetRival/img", exist_ok=True)
    os.makedirs("tect/TetRival/bgm", exist_ok=True)
    os.makedirs("tect/TetRival/se", exist_ok=True)
    
    app = QApplication(sys.argv)
    rival_app = TetrisApp()
    rival_app.show()
    sys.exit(app.exec())
