import sys
import random
import os
import time # スムーズなアニメーションのために追加
from collections import Counter
import matplotlib.pyplot as plt
from pathlib import Path
from PySide6.QtWidgets import ( # PyQt6からPySide6に変更
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout, QGridLayout,
    QComboBox, QPushButton, QLabel, QTextBrowser, QMessageBox, QDialog,
    QDialogButtonBox, QListWidget, QAbstractItemView, QSpinBox,
    QTableWidget, QTableWidgetItem, QStatusBar, QStackedWidget, QInputDialog,QListWidgetItem,
    QFrame, QLineEdit, QGraphicsView, QGraphicsScene, QGraphicsTextItem, QSlider
)
from PySide6.QtGui import QPixmap, QColor, QPainter, QLinearGradient, QKeySequence,QBrush, QFont,QIcon # PyQt6からPySide6に変更
from PySide6.QtCore import Qt, QTimer, QRectF, QPropertyAnimation, QEasingCurve, QPointF, QParallelAnimationGroup # PyQt6からPySide6に変更

# オーディオ再生のためにpygameを使用
try:
    import pygame
    PYGAME_AVAILABLE = True
except ImportError:
    PYGAME_AVAILABLE = False
    print("[警告] Pygameライブラリが見つかりません。オーディオは再生されません。")


# --- アセットパス定義関数 ---
def resource_path(relative_path):
    """ アセットへの絶対パスを生成します。PyInstallerやインタラクティブ環境に対応。 """
    try:
        # PyInstallerによって作成された一時フォルダからパスを取得
        base_path = sys._MEIPASS
    except AttributeError:
        # 通常のPython環境では、このスクリプトファイルがあるディレクトリを基準にする
        try:
            base_path = os.path.dirname(os.path.abspath(__file__))
        except NameError:
            # __file__が未定義の場合、カレントワーキングディレクトリを基準にする
            base_path = os.path.abspath(".")
    
    return os.path.join(base_path, relative_path)

# --- アセットパス ---
ASSET_DIR = "tect"
# --- ロゴ ---
APP_ICON_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "Civcicon.png"))
GCS2_LOGO_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "GCS2.png"))
SAPOCREATE_LOGO_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "SAPOCREATE.png"))
GCOT_LOGO_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "GCOT.png"))
# --- 背景画像 ---
TITLE_BACKGROUND_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "Civic_Strategy.png"))
CREDITS_BACKGROUND_PATH = resource_path(os.path.join(ASSET_DIR, "titlepng", "Civic_ending.png"))
# --- BGM ---
TITLE_BGM_PATH = resource_path(os.path.join(ASSET_DIR, "bgm", "442_long_BPM160.mp3"))
GAME_BGM_PATH = resource_path(os.path.join(ASSET_DIR, "bgm", "328_long_BPM86.mp3"))
CREDITS_BGM_PATH = resource_path(os.path.join(ASSET_DIR, "bgm", "231_long_BPM130.mp3"))

# ----------------------------------------------------
# Part 1: データ構造の定義
# ----------------------------------------------------
class Party:
    def __init__(self, name, funds):
        self.name = name; self.funds = funds; self.candidates = []
        self.coalition_partners = []; self.regional_allies = []; self.color = 'gold'
class Candidate:
    def __init__(self, name, assigned_region):
        self.name = name; self.assigned_region = assigned_region
class RivalParty:
    def __init__(self, name, rank, specialty, owner='cpu'):
        self.name = name; self.rank = rank; self.specialty = specialty; self.owner = owner
        if rank == 'A': self.support_range = (5, 10); self.coalition_probability = 5
        elif rank == 'B': self.support_range = (3, 8); self.coalition_probability = 10
        else: self.support_range = (1, 5); self.coalition_probability = 15
class Region:
    def __init__(self, name, seats, population, interests, age_demographics, area_type):
        self.name = name; self.seats = seats; self.population = population; self.interests = interests
        self.age_demographics = age_demographics; self.area_type = area_type; self.support_rates = {}
    def get_votes_per_party(self): return {p: int(self.population * (r / 100)) for p, r in self.support_rates.items()}
class Policy:
    def __init__(self, name, category, appeal_point):
        self.name = name; self.category = category; self.appeal_point = appeal_point

# ---------------------------------------------------------
# Part 2: ダイアログクラス
# ---------------------------------------------------------
class RecruitDialog(QDialog):
    def __init__(self, party, all_regions, party_candidates_counts, parent=None):
        super().__init__(parent); self.setWindowTitle("候補者の擁立 (1人あたり1000万円)"); self.party = party; self.all_regions = all_regions; self.party_candidates_counts = party_candidates_counts; self.results = []
        layout = QVBoxLayout(self); self.info_label = QLabel(f"現在の党資金: {self.party.funds}万円\n各選挙区に擁立する人数を入力してください。"); layout.addWidget(self.info_label); self.table = QTableWidget(); self.table.setColumnCount(4); self.table.setHorizontalHeaderLabels(["選挙区", "定数", "現擁立数", "今回擁立する人数"])
        for i, region_name in enumerate(self.all_regions.keys()):
            region = self.all_regions[region_name]; current_candidates = self.party_candidates_counts.get(region_name, 0); self.table.insertRow(i); self.table.setItem(i, 0, QTableWidgetItem(region_name)); self.table.setItem(i, 1, QTableWidgetItem(str(region.seats))); self.table.setItem(i, 2, QTableWidgetItem(str(current_candidates))); spin_box = QSpinBox(); spin_box.setMinimum(0); max_new = region.seats - current_candidates; spin_box.setMaximum(max_new if max_new > 0 else 0); self.table.setCellWidget(i, 3, spin_box)
            for j in range(3): self.table.item(i, j).setTextAlignment(Qt.AlignmentFlag.AlignCenter)
        self.table.resizeColumnsToContents(); layout.addWidget(self.table)
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel); button_box.accepted.connect(self.accept_selection); button_box.rejected.connect(self.reject); layout.addWidget(button_box)
    def accept_selection(self):
        total_cost = 0; self.results = []
        for i in range(self.table.rowCount()):
            count = self.table.cellWidget(i, 3).value()
            if count > 0: region_name = self.table.item(i, 0).text(); self.results.append({'region': region_name, 'count': count}); total_cost += count * 1000
        if total_cost == 0: QMessageBox.information(self, "確認", "擁立する候補者はいません。"); self.accept(); return
        if self.party.funds < total_cost: QMessageBox.warning(self, "資金不足", f"合計{total_cost}万円が必要ですが、資金が足りません。"); return
        self.accept()
class CoalitionDialog(QDialog):
    def __init__(self, negotiable_parties, parent=None):
        super().__init__(parent); self.setWindowTitle("連立交渉"); self.selected_rival = None; layout = QVBoxLayout(self); layout.addWidget(QLabel("交渉相手を選択してください。\n交渉には3日を要します。")); self.party_list = QListWidget()
        for rival, prob, penalty in negotiable_parties:
            penalty_text = f" (議席不利: {penalty}%)" if penalty < 0 else ""; item_text = f"{rival.name} (クラス{rival.rank}) - 現在の連立可能性: {prob}%{penalty_text}"; item = QListWidgetItem(item_text); item.setData(Qt.ItemDataRole.UserRole, rival); self.party_list.addItem(item)
        layout.addWidget(self.party_list); button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel); button_box.accepted.connect(self.accept_selection); button_box.rejected.connect(self.reject); layout.addWidget(button_box)
    def accept_selection(self):
        selected_item = self.party_list.currentItem()
        if not selected_item: QMessageBox.warning(self, "エラー", "交渉相手を選択してください。"); return
        self.selected_rival = selected_item.data(Qt.ItemDataRole.UserRole); self.accept()
class OrganizePartyDialog(QDialog):
    def __init__(self, organizable_regions, parent=None):
        super().__init__(parent); self.setWindowTitle("地域政党の組織 (コスト: 5000万)"); self.selected_region = None; layout = QVBoxLayout(self); layout.addWidget(QLabel("組織する選挙区を選択してください。")); self.region_list = QListWidget(); self.region_list.addItems(organizable_regions); layout.addWidget(self.region_list)
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel); button_box.accepted.connect(self.accept_selection); button_box.rejected.connect(self.reject); layout.addWidget(button_box)
    def accept_selection(self):
        selected_item = self.region_list.currentItem()
        if not selected_item: QMessageBox.warning(self, "エラー", "選挙区を選択してください。"); return
        self.selected_region = selected_item.text(); self.accept()
class ResultDialog(QDialog):
    def __init__(self, title, report_text, chart_path, parent=None):
        super().__init__(parent); self.setWindowTitle(title); self.setMinimumSize(800, 700); layout = QVBoxLayout(self); text_browser = QTextBrowser(); text_browser.setPlainText(report_text); layout.addWidget(text_browser, 1)
        if chart_path and os.path.exists(chart_path):
            chart_label = QLabel(); pixmap = QPixmap(chart_path); chart_label.setPixmap(pixmap.scaled(750, 600, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)); chart_label.setAlignment(Qt.AlignmentFlag.AlignCenter); layout.addWidget(chart_label, 2)
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok); button_box.accepted.connect(self.accept); layout.addWidget(button_box)
class FoundAllyDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent); self.setWindowTitle("系列政党の設立"); self.selected_class = None; self.party_name = None; self.cost = 0; self.costs = {'A': 30000, 'B': 20000, 'C': 10000}; layout = QVBoxLayout(self); layout.addWidget(QLabel("設立する系列政党のクラスを選択してください。")); self.class_combo = QComboBox()
        for rank, cost in self.costs.items(): self.class_combo.addItem(f"クラス {rank} ({cost:,}万円)", rank)
        self.class_combo.currentTextChanged.connect(self.update_cost); layout.addWidget(self.class_combo); layout.addWidget(QLabel("系列政党の名前を入力してください。")); self.name_edit = QLineEdit("新市民連合"); layout.addWidget(self.name_edit); self.info_label = QLabel(); self.update_cost(); layout.addWidget(self.info_label)
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel); button_box.accepted.connect(self.accept_creation); button_box.rejected.connect(self.reject); layout.addWidget(button_box)
    def update_cost(self): self.cost = self.costs[self.class_combo.currentData()]; self.info_label.setText(f"設立コスト: {self.cost:,}万円")
    def accept_creation(self):
        self.selected_class = self.class_combo.currentData(); self.party_name = self.name_edit.text().strip()
        if not self.party_name: QMessageBox.warning(self, "入力エラー", "政党名を入力してください。"); return
        self.accept()

class MapWidget(QWidget):
    def __init__(self, all_region_names, parent=None):
        super().__init__(parent); self.region_labels = {}; self.init_ui(all_region_names)
    def init_ui(self, all_region_names):
        layout = QGridLayout(self); layout.setSpacing(5); row, col = 0, 0
        for region_name in sorted(all_region_names):
            label = QLabel(region_name); label.setAlignment(Qt.AlignmentFlag.AlignCenter); label.setMinimumSize(110, 45); label.setFrameShape(QFrame.Shape.StyledPanel); label.setStyleSheet("background-color: #E0E0E0; color: black; font-weight: bold; border-radius: 4px;"); layout.addWidget(label, row, col); self.region_labels[region_name] = label
            col += 1
            if col >= 5: col = 0; row += 1
    def update_map(self, regions, party_colors_map, coalition_party_names):
        for region_name, region_data in regions.items():
            if region_name in self.region_labels:
                label = self.region_labels[region_name]
                if not region_data.support_rates: top_party_name = None
                else: top_party_name = max(region_data.support_rates, key=region_data.support_rates.get)
                bg_color_hex = party_colors_map.get(top_party_name, "#E0E0E0"); q_color = QColor(bg_color_hex); text_color = "black" if q_color.lightness() > 128 else "white"
                if top_party_name in coalition_party_names: border_style = "border: 3px solid gold;"
                else: border_style = "border: 1px solid #555;"
                label.setStyleSheet(f"background-color: {bg_color_hex}; color: {text_color}; font-weight: bold; {border_style} border-radius: 4px;")
                tooltip_text = f"<b>{region_name} (定数: {region_data.seats})</b><br>"
                sorted_rates = sorted(region_data.support_rates.items(), key=lambda i: i[1], reverse=True)
                for i, (party, rate) in enumerate(sorted_rates[:3]):
                    coalition_mark = " (連合)" if party in coalition_party_names else ""; tooltip_text += f"{i+1}. {party}: {rate}%{coalition_mark}<br>"
                label.setToolTip(tooltip_text)

class TitleScreen(QWidget):
    def __init__(self, app_instance, parent=None):
        super().__init__(parent); self.app_instance = app_instance; self.init_ui()
    def paintEvent(self, event):
        painter = QPainter(self)
        pixmap = QPixmap(TITLE_BACKGROUND_PATH)
        if not pixmap.isNull():
            scaled_pixmap = pixmap.scaled(self.size(), Qt.AspectRatioMode.KeepAspectRatioByExpanding, Qt.TransformationMode.SmoothTransformation)
            painter.drawPixmap(self.rect(), scaled_pixmap)
        else:
            gradient = QLinearGradient(0, 0, 0, self.height())
            gradient.setColorAt(0.0, QColor("#3D81EC")); gradient.setColorAt(1.0, QColor("#0C56D4"))
            painter.fillRect(self.rect(), gradient)
        super().paintEvent(event)
    def init_ui(self):
        layout = QVBoxLayout(self); layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title_label = QLabel("Game of Civic Strategy"); title_label.setStyleSheet("font-size: 64px; font-family: 'Georgia', 'Times New Roman', serif; font-weight: bold; color: white; text-shadow: 3px 3px 8px black;")
        sub_title_label = QLabel("- National Election Edition -"); sub_title_label.setStyleSheet("font-size: 28px; font-family: 'Georgia', 'Times New Roman', serif; color: #E0E8F0; margin-bottom: 50px; text-shadow: 2px 2px 4px black;")
        btn_style = """ QPushButton { font-size: 22px; font-family: "MS PGothic", "Meiryo UI", sans-serif; font-weight: bold; color: black; min-width: 320px; min-height: 55px; border: 1px solid #003C74; border-radius: 6px; background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #FFFFFF, stop:1 #E5F1FB); } QPushButton:hover { background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #FFFFFF, stop:1 #D0E8FA); border: 1px solid #3C7FB1; } QPushButton:pressed { background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #C0D8F0, stop:1 #E6F1FB); border-style: inset; padding-top: 2px; } """
        btn_start = QPushButton("ゲーム開始 (S)"); btn_start.setStyleSheet(btn_style); btn_start.clicked.connect(self.app_instance.show_game_screen); btn_start.setShortcut(QKeySequence("S"))
        btn_credits = QPushButton("スタッフロール"); btn_credits.setStyleSheet(btn_style); btn_credits.clicked.connect(self.app_instance.show_credits_screen)
        btn_quit = QPushButton("終了 (Q)"); btn_quit.setStyleSheet(btn_style); btn_quit.clicked.connect(QApplication.instance().quit); btn_quit.setShortcut(QKeySequence("Q"))
        layout.addStretch(1); layout.addWidget(title_label, alignment=Qt.AlignmentFlag.AlignCenter); layout.addWidget(sub_title_label, alignment=Qt.AlignmentFlag.AlignCenter); layout.addWidget(btn_start, alignment=Qt.AlignmentFlag.AlignCenter); layout.addSpacing(10); layout.addWidget(btn_credits, alignment=Qt.AlignmentFlag.AlignCenter); layout.addSpacing(10); layout.addWidget(btn_quit, alignment=Qt.AlignmentFlag.AlignCenter); layout.addStretch(2)

# --- ここからが最新の改善版 CreditsScreen クラス ---
class CreditsScreen(QWidget):
    def __init__(self, app_instance, parent=None):
        super().__init__(parent)
        self.app_instance = app_instance
        self.text_item = None

        # メインのスクロールタイマー
        self.scroll_timer = QTimer(self)
        self.scroll_timer.timeout.connect(self.scroll_step)
        self.scroll_speed_pps = 40  # 1秒間にスクロールするピクセル数
        self.last_update_time = 0

        self.init_ui()

    def paintEvent(self, event):
        painter = QPainter(self)
        pixmap = QPixmap(CREDITS_BACKGROUND_PATH)
        if not pixmap.isNull():
            scaled_pixmap = pixmap.scaled(self.size(), Qt.AspectRatioMode.KeepAspectRatioByExpanding, Qt.TransformationMode.SmoothTransformation)
            painter.drawPixmap(self.rect(), scaled_pixmap)
        else:
            painter.fillRect(self.rect(), QColor("black"))
        super().paintEvent(event)

    def init_ui(self):
        self.setStyleSheet("background-color: transparent;")
        
        # --- メインレイアウトを水平分割に変更 ---
        main_layout = QHBoxLayout(self)
        main_layout.setContentsMargins(0,0,0,0)
        main_layout.setSpacing(0)

        # --- 左パネル（スクロールクレジット用）---
        left_pane = QWidget()
        left_layout = QVBoxLayout(left_pane)
        self.scene = QGraphicsScene()
        self.view = QGraphicsView(self.scene)
        self.view.setStyleSheet("border: none; background-color: transparent;")
        self.scene.setBackgroundBrush(QBrush(Qt.GlobalColor.transparent))
        self.view.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        self.view.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        self.view.setViewportUpdateMode(QGraphicsView.ViewportUpdateMode.FullViewportUpdate)
        left_layout.addWidget(self.view)

        # --- 右パネル（固定表示用）---
        right_pane = QWidget()
        right_layout = QVBoxLayout(right_pane)
        right_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        # アイコン
        icon_label = QLabel()
        if os.path.exists(APP_ICON_PATH):
            pixmap = QPixmap(APP_ICON_PATH)
            icon_label.setPixmap(pixmap.scaled(256, 256, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation))
        icon_label.setAlignment(Qt.AlignmentFlag.AlignCenter)

        # タイトル
        title_label = QLabel("Game of Civic Strategy")
        title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title_label.setStyleSheet("font-size: 48px; font-family: 'Georgia', 'Times New Roman', serif; font-weight: bold; color: white; text-shadow: 2px 2px 8px black;")
        title_label.setWordWrap(True)

        # 謝辞
        thanks_label = QLabel("Thank you for playing!!")
        thanks_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        thanks_label.setStyleSheet("font-size: 24px; font-family: 'Georgia', 'Times New Roman', serif; color: #E0E8F0; margin-top: 20px;")

        right_layout.addStretch(1)
        right_layout.addWidget(icon_label)
        right_layout.addWidget(title_label)
        right_layout.addWidget(thanks_label)
        right_layout.addStretch(2)
        
        # メインレイアウトに左右のパネルを追加（左:右 = 5:4 の比率）
        main_layout.addWidget(left_pane, 5)
        main_layout.addWidget(right_pane, 4)


    def start_scrolling(self):
        self.stop_scrolling()
        self.prepare_credits()
        if not self.text_item:
            return
        
        self.last_update_time = time.perf_counter()
        self.scroll_timer.start(16) # 約60fps


    def stop_scrolling(self):
        self.scroll_timer.stop()

    def scroll_step(self):
        # 経過時間に基づいてスクロール量を計算
        current_time = time.perf_counter()
        delta_time = current_time - self.last_update_time
        self.last_update_time = current_time
        scroll_amount = self.scroll_speed_pps * delta_time

        # クレジットテキストのY座標を更新
        new_y = self.text_item.y() - scroll_amount
        self.text_item.setY(new_y)

        # クレジットが完全に画面外に出たら終了
        if (self.text_item.y() + self.text_item.boundingRect().height()) < 0:
            self.return_to_title()

    def return_to_title(self):
        self.stop_scrolling()
        self.app_instance.show_title_screen()

    def prepare_credits(self):
        self.scene.clear()
        self.text_item = None
        
        # ビューのサイズに合わせてシーンの矩形を設定
        self.scene.setSceneRect(QRectF(self.view.rect()))

        gcs2_logo_html = Path(GCS2_LOGO_PATH).as_uri() if os.path.exists(GCS2_LOGO_PATH) else ""
        sapocreate_logo_html = Path(SAPOCREATE_LOGO_PATH).as_uri() if os.path.exists(SAPOCREATE_LOGO_PATH) else ""
        gcot_logo_html = Path(GCOT_LOGO_PATH).as_uri() if os.path.exists(GCOT_LOGO_PATH) else ""

        # --- スクロールするクレジット部分 ---
        credits_text = f"""
        <div style='width: 450px; font-family: "Georgia", "Times New Roman", serif; color: white; font-size: 16pt;'>
        <h2 align="center" style='font-size: 22pt; font-weight: bold;'>開発者</h2>
        <p align="center" style='font-size: 18pt;'>SECA Project and Sapo Create</p>
        <p align="center">GCS-97 サポ / from Kagawa</p>
        <p align="center">GCS-97 initial members (joined in 2023)</p>
        <p align="center"><span style="background-color: #005588; padding: 3px 8px; border-radius: 5px; font-size: 14pt;">Sub-admin of GCS-97</span></p>
        <br><br>
        
        <h2 align="center" style='font-size: 22pt; font-weight: bold;'>デバッカー</h2>
        <p align="center" style='font-size: 18pt;'>GCOT supported by GCS-97</p>
        <p align="center">GCS-97 Sayuri / from Kagawa</p>
        <p align="center">GCS-97 initial members (joined in 2023)</p>
        <p align="center"><span style="background-color: #FF0000; padding: 3px 8px; border-radius: 5px; font-size: 14pt;">admin of GCS-97</span></p>
        <br><br>
        
        <h2 align="center" style='font-size: 22pt; font-weight: bold;'>使用音源</h2>
        <p align="center">待機画面BGM: "笑顔ある厨房"</p>
        <p align="center">プレイ画面BGM: "コーヒーの淹れ方"</p>
        <p align="center">スタッフロールBGM: "ピーチコーミング"</p>
        <p align="center" style='font-size: 14pt;'>制作: BGMer (https://bgmer.net)</p>
        <br><br>
        
        <h2 align="center" style='font-size: 22pt; font-weight: bold;'>開発・提供チーム</h2>
        <p align="center"><img src="{gcs2_logo_html}" width="200"></p>
        <h3 align="center">GCS-97(Global Create Station)</h3>
        <h3 align="center">Established in 2023</h3>
        <br><br>
        <p align="center"><img src="{sapocreate_logo_html}" width="200"></p>
        <h3 align="center">Sapo Create</h3>
        <br><br>
        <p align="center"><img src="{gcot_logo_html}" width="200"></p>
        <h3 align="center">GCOT supported by GCS-97</h3>
        <br><br>
        
        <h2 align="center" style='font-size: 26pt; font-weight: bold;'>Special Thanks</h2>
        <p align="center">Google Gemini</p>
        <p align="center">PySide Project</p>
        <p align="center">Pygame Project</p>
        <p align="center">matplotlib Project</p>
        <p align="center">To all the players who played with us.</p>
        <br><br>
        
        <h3 align="center" style='font-size: 14pt;'>Copyright © 2025 GCS-97/(SAPOCreate,GCOT). <br>All Rights Reserved.</h3>
        <br><br><br><br>
        </div>
        """
        self.text_item = QGraphicsTextItem()
        self.text_item.setHtml(credits_text)
        # 左パネルの中央に配置
        text_x = (self.view.width() - self.text_item.boundingRect().width()) / 2
        # 初期位置は画面下
        self.text_item.setPos(text_x, self.view.height())
        self.scene.addItem(self.text_item)


    def resizeEvent(self, event):
        super().resizeEvent(event)
        is_running = self.scroll_timer.isActive()
        self.stop_scrolling()
        self.prepare_credits()
        if is_running:
             self.start_scrolling()

    def keyPressEvent(self, event):
        if event.key() == Qt.Key.Key_Escape:
            self.return_to_title()
        super().keyPressEvent(event)

    def mousePressEvent(self, event):
        self.return_to_title()
        super().mousePressEvent(event)
# --- 改善版 CreditsScreen クラスはここまで ---


# ---------------------------------------------------------
# Part 3: メイン画面クラス
# ---------------------------------------------------------
class GameScreen(QWidget):
    def __init__(self, app_instance, parent=None):
        super().__init__(parent); self.app_instance = app_instance; self.init_master_data()
        self.party = None; self.days_left = 0; self.tv_speech_left = 0; self.active_rivals = []; self.active_parties = []; self.ai_coalitions = []
        self.setup_ui()
    def start_game(self): self.reset_game()
    def reset_game(self):
        party_name, ok = QInputDialog.getText(self, "政党名の設定", "あなたの党の名前を入力してください:", text="市民戦略党")
        if not ok or not party_name.strip(): party_name = "市民戦略党"
        color_names = list(self.available_colors.keys()); color_name, ok = QInputDialog.getItem(self, "政党カラーの設定", "あなたの党のシンボルカラーを選択してください:", color_names, 0, False)
        if not ok or not color_name: color_name = "ゴールド"
        party_color_code = self.available_colors[color_name]
        self.party = Party(name=party_name, funds=20000); self.party.color = party_color_code; self.tv_speech_left = 10; self.ai_coalitions = []
        self.init_master_data()
        for rival in self.all_rival_parties: rival.__init__(rival.name, rival.rank, rival.specialty)
        self.active_rivals = random.sample([p for p in self.all_rival_parties if p.owner == 'cpu'], 8)
        self.active_parties = [self.party.name] + [r.name for r in self.active_rivals]
        self.initialize_support_rates(); self.days_left = 65; self.update_all_displays()
        rival_names = '」、「'.join([r.name for r in self.active_rivals])
        QMessageBox.information(self, "選挙戦開始！", f"「{self.party.name}」の戦いが始まる！\n今回のライバルは「{rival_names}」の8党です！")
    def init_master_data(self):
        self.all_rival_parties = [ RivalParty("緑化党", 'A', {'type': 'policy', 'category': '環境'}), RivalParty("立憲主義党", 'B', {'type': 'policy', 'category': '交通'}), RivalParty("若年層支援党", 'B', {'type': 'speech', 'theme': 'youth'}), RivalParty("村の民団結党", 'C', {'type': 'region_type', 'area': '村'}), RivalParty("京保護党", 'C', {'type': 'region_specific', 'region': '北陸京'}), RivalParty("立憲環境保護党", 'B', {'type': 'region_type', 'area': '町'}), RivalParty("中川村保守党", 'A', {'type': 'limited_activity', 'regions': ['中川村', '小金村']}), RivalParty("共和党", 'A', {'type': 'high_seats'}), RivalParty("町議会議員連合", 'B', {'type': 'region_type', 'area': '町'}), RivalParty("自由市民ネットワーク", 'B', {'type': 'region_type', 'area': '町'}), RivalParty("中大津村の党", 'C', {'type': 'region_specific', 'region': '中大津村'}), RivalParty("大島村の党", 'C', {'type': 'region_specific', 'region': '大島村'}), RivalParty("小金村の党", 'C', {'type': 'region_specific', 'region': '小金村'}), RivalParty("中川村の党", 'C', {'type': 'region_specific', 'region': '中川村'}), RivalParty("太目村の党", 'C', {'type': 'region_specific', 'region': '太目村'})]
        self.regions = { "長野市": Region("長野市", 7, 370000, {"経済": 0.8, "福祉": 0.5, "若者支援": 0.7}, {"youth": 0.4, "middle": 0.5, "senior": 0.1}, "都市"), "北陸京": Region("北陸京", 15, 1500000,{"経済": 0.9, "若者支援": 0.8, "交通": 0.7}, {"youth": 0.5, "middle": 0.4, "senior": 0.1}, "都市"), "南都": Region("南都", 12, 1200000, {"経済": 0.8, "若者支援": 0.9, "交通": 0.6}, {"youth": 0.6, "middle": 0.3, "senior": 0.1}, "都市"), "東中野町": Region("東中野町", 2, 80000, {"経済": 0.5, "福祉": 0.8, "交通": 0.6}, {"youth": 0.2, "middle": 0.6, "senior": 0.2}, "町"), "西中野町": Region("西中野町", 4, 120000, {"経済": 0.6, "福祉": 0.7, "交通": 0.7}, {"youth": 0.3, "middle": 0.5, "senior": 0.2}, "町"), "萩原町": Region("萩原町", 2, 60000, {"経済": 0.4, "福祉": 0.9, "地域振興": 0.8}, {"youth": 0.1, "middle": 0.5, "senior": 0.4}, "町"), "郡家町": Region("郡家町", 2, 55000, {"交通": 0.8, "地域振興": 0.7, "福祉": 0.6}, {"youth": 0.2, "middle": 0.5, "senior": 0.3}, "町"), "境町": Region("境町", 3, 95000, {"経済": 0.7, "交通": 0.8, "福祉": 0.5}, {"youth": 0.3, "middle": 0.6, "senior": 0.1}, "町"), "港町": Region("港町", 3, 110000, {"経済": 0.9, "交通": 0.7, "地域振興": 0.5}, {"youth": 0.4, "middle": 0.5, "senior": 0.1}, "町"), "山手町": Region("山手町", 2, 75000, {"福祉": 0.8, "若者支援": 0.7, "経済": 0.5}, {"youth": 0.3, "middle": 0.4, "senior": 0.3}, "町"), "中大津村": Region("中大津村", 1, 20000, {"福祉": 0.8, "地域振興": 0.9, "交通": 0.5}, {"youth": 0.1, "middle": 0.4, "senior": 0.5}, "村"), "大島村": Region("大島村", 1, 15000, {"福祉": 0.7, "地域振興": 1.0, "経済": 0.3}, {"youth": 0.1, "middle": 0.3, "senior": 0.6}, "村"), "小金村": Region("小金村", 1, 18000, {"経済": 0.6, "地域振興": 0.9, "福祉": 0.7}, {"youth": 0.2, "middle": 0.4, "senior": 0.4}, "村"), "中川村": Region("中川村", 1, 22000, {"福祉": 0.9, "地域振興": 0.8, "経済": 0.5}, {"youth": 0.1, "middle": 0.3, "senior": 0.6}, "村"), "太目村": Region("太目村", 1, 12000, {"地域振興": 1.0, "福祉": 0.8, "交通": 0.4}, {"youth": 0.1, "middle": 0.2, "senior": 0.7}, "村"), "下田村": Region("下田村", 1, 19000, {"地域振興": 0.9, "福祉": 0.9, "交通": 0.3}, {"youth": 0.1, "middle": 0.3, "senior": 0.6}, "村"), "上田村": Region("上田村", 1, 25000, {"経済": 0.5, "地域振興": 0.8, "福祉": 0.8}, {"youth": 0.2, "middle": 0.4, "senior": 0.4}, "村"), "東山村": Region("東山村", 1, 14000, {"福祉": 1.0, "地域振興": 0.7, "経済": 0.2}, {"youth": 0.1, "middle": 0.2, "senior": 0.7}, "村"), "西山村": Region("西山村", 1, 16000, {"交通": 0.6, "地域振興": 0.9, "福祉": 0.7}, {"youth": 0.1, "middle": 0.4, "senior": 0.5}, "村"), "南川村": Region("南川村", 1, 21000, {"若者支援": 0.3, "地域振興": 0.8, "福祉": 0.9}, {"youth": 0.2, "middle": 0.3, "senior": 0.5}, "村"), }
        self.policies = [Policy("大規模減税", "経済", 10), Policy("公共事業の拡大", "経済", 8), Policy("子育て支援拡充", "福祉", 12), Policy("高齢者医療の無料化", "福祉", 10), Policy("若者向け雇用の創出", "若者支援", 11), Policy("大学授業料の減免", "若者支援", 9), Policy("コミュニティバスの運行", "交通", 9), Policy("特産品ブランド化支援", "地域振興", 12)]
        self.actions = {"政策発表": 300, "街頭演説": 150, "チラシ配り": 100, "テレビ演説": 1000, "何もしない": 0}
        self.speech_themes = ["若者向け演説", "現役世代向け演説", "高齢者向け演説"]
        self.total_seats = sum(r.seats for r in self.regions.values())
        self.majority_threshold = self.total_seats // 2 + 1
        self.available_colors = {"ゴールド": "gold", "赤": "red", "青": "blue", "緑": "green", "オレンジ": "orange", "紫": "purple", "ピンク": "pink", "水色": "skyblue", "茶色": "brown"}
    def initialize_support_rates(self):
        num_parties = len(self.active_parties); base_support = 100 // num_parties
        for r in self.regions.values():
            r.support_rates = {p: base_support for p in self.active_parties}
            rem = 100 % num_parties
            if rem > 0: r.support_rates[self.active_parties[0]] += rem
            for rival in self.active_rivals:
                bonus = 0; specialty = rival.specialty
                if specialty['type'] == 'region_type' and r.area_type == specialty['area']:
                    if rival.name == "村の民団結党": bonus = 15
                    if rival.name in ["立憲環境保護党", "町議会議員連合", "自由市民ネットワーク"]: bonus = 10
                elif specialty['type'] == 'region_specific' and r.name == specialty['region']: bonus = 15
                elif specialty['type'] == 'limited_activity' and r.name in specialty['regions']: bonus = 20
                if bonus > 0:
                    other_parties = [p for p in self.active_parties if p != rival.name]
                    if not other_parties: continue
                    loss_per_party = bonus // len(other_parties)
                    for party_name in other_parties: r.support_rates[party_name] -= loss_per_party
                    r.support_rates[rival.name] += bonus
            current_total = sum(r.support_rates.values())
            if current_total != 100: r.support_rates[self.active_parties[0]] += (100 - current_total)
    def setup_ui(self):
        main_layout = QHBoxLayout(self); left_panel = QWidget(); left_layout = QVBoxLayout(left_panel); main_layout.addWidget(left_panel, 2); right_panel = QWidget(); right_layout = QVBoxLayout(right_panel); main_layout.addWidget(right_panel, 3); self.status_label=QLabel(); left_layout.addWidget(self.status_label)
        self.recruit_button=QPushButton("候補者を擁立する"); self.recruit_button.clicked.connect(self.recruit_candidate_action); left_layout.addWidget(self.recruit_button)
        self.organize_button = QPushButton("地域政党を組織 (5000万)"); self.organize_button.clicked.connect(self.open_organize_dialog); left_layout.addWidget(self.organize_button)
        self.found_ally_button = QPushButton("系列政党を設立"); self.found_ally_button.clicked.connect(self.open_found_ally_dialog); left_layout.addWidget(self.found_ally_button)
        self.coalition_button = QPushButton("連立交渉を行う (3日消費)"); self.coalition_button.clicked.connect(self.open_coalition_dialog); left_layout.addWidget(self.coalition_button); left_layout.addSpacing(10)
        left_layout.addWidget(QLabel("① 操作する選挙区"));self.region_combo=QComboBox(); self.region_combo.currentTextChanged.connect(self.update_info_display_on_demand); left_layout.addWidget(self.region_combo);left_layout.addWidget(QLabel("② 行動を選択"));self.action_combo=QComboBox(); self.action_combo.currentTextChanged.connect(self.update_detail_combo); left_layout.addWidget(self.action_combo);left_layout.addWidget(QLabel("③ 詳細を選択"));self.detail_combo=QComboBox(); left_layout.addWidget(self.detail_combo)
        self.execute_button=QPushButton("行動実行 (1日消費)"); self.execute_button.clicked.connect(self.execute_turn); left_layout.addWidget(self.execute_button);left_layout.addStretch()
        btn_back_to_title = QPushButton("タイトルへ戻る"); btn_back_to_title.clicked.connect(self.confirm_return_to_title); left_layout.addWidget(btn_back_to_title)
        tab_button_layout = QHBoxLayout(); self.info_button = QPushButton("情報"); self.map_button = QPushButton("マップ"); tab_button_layout.addWidget(self.info_button); tab_button_layout.addWidget(self.map_button); tab_button_layout.addStretch()
        right_layout.addLayout(tab_button_layout); self.right_stacked_widget = QStackedWidget(); right_layout.addWidget(self.right_stacked_widget)
        self.info_display = QTextBrowser(); self.right_stacked_widget.addWidget(self.info_display)
        self.map_widget = MapWidget(self.regions.keys()); self.right_stacked_widget.addWidget(self.map_widget)
        self.info_button.clicked.connect(lambda: self.right_stacked_widget.setCurrentWidget(self.info_display)); self.map_button.clicked.connect(lambda: self.right_stacked_widget.setCurrentWidget(self.map_widget)); self.right_stacked_widget.setCurrentWidget(self.info_display)
    def confirm_return_to_title(self):
        reply = QMessageBox.question(self, '確認', "現在のゲームを終了し、タイトル画面に戻りますか？", QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No, QMessageBox.StandardButton.No)
        if reply == QMessageBox.StandardButton.Yes:
            self.app_instance.show_title_screen()
    def get_party_colors_map(self):
        rival_colors = { '緑化党': 'green', '立憲主義党': 'navy', '若年層支援党': 'skyblue', '村の民団結党': 'brown', '京保護党':'purple', '立憲環境保護党':'teal', '中川村保守党':'darkred', "共和党": "orangered", "町議会議員連合": "saddlebrown", "自由市民ネットワーク": "cyan", "中大津村の党": 'olive', "大島村の党": 'peru', "小金村の党": 'khaki', "中川村の党": 'sienna', "太目村の党": 'darkseagreen', '系列地域政党': 'lightcoral'}
        party_colors = rival_colors.copy()
        if not self.party: return party_colors
        player_color = self.party.color; party_colors[self.party.name] = player_color
        for partner_name in self.party.coalition_partners: party_colors[partner_name] = player_color
        for ally in self.party.regional_allies: party_colors[ally.name] = player_color
        for coalition in self.ai_coalitions:
            if coalition: representative = list(coalition)[0]; coalition_color = rival_colors.get(representative, 'gray'); 
            for member in coalition: party_colors[member] = coalition_color
        ai_regional_allies = [p for p in self.active_rivals if p.owner in [r.name for r in self.active_rivals]]
        for ally in ai_regional_allies: parent_color = party_colors.get(ally.owner, 'gray'); party_colors[ally.name] = parent_color
        return party_colors
    def open_found_ally_dialog(self):
        dialog = FoundAllyDialog(self)
        if dialog.exec():
            cost = dialog.cost; rank = dialog.selected_class; name = dialog.party_name
            if self.party.funds < cost: QMessageBox.warning(self, "資金不足", f"設立コストが足りません。(必要: {cost:,}万円)"); return
            if name in self.active_parties: QMessageBox.warning(self, "名前の重複", "その政党名はすでに存在します。"); return
            self.execute_found_ally(name, rank, cost)
    def execute_found_ally(self, name, rank, cost):
        self.party.funds -= cost; specialty = {'type': 'nationwide_ally', 'rank': rank}
        new_ally = RivalParty(name, rank, specialty, owner=self.party.name)
        self.party.regional_allies.append(new_ally); self.active_rivals.append(new_ally); self.active_parties.append(new_ally.name)
        for region in self.regions.values():
            region.support_rates[new_ally.name] = 0
            if region.support_rates[self.party.name] > 0: region.support_rates[self.party.name] -= 1; region.support_rates[new_ally.name] += 1
        QMessageBox.information(self, "設立完了", f"全国系列政党「{name}」(クラス{rank})が設立されました！"); self.update_all_displays()
    def recruit_candidate_action(self):
        party_candidates_counts = Counter(c.assigned_region for c in self.party.candidates); dialog = RecruitDialog(self.party, self.regions, party_candidates_counts, self)
        if dialog.exec() and dialog.results:
            total_cost = 0; recruited_info = []
            for result in dialog.results:
                region_name = result['region']; count = result['count']; total_cost += count * 1000
                for _ in range(count): self.party.candidates.append(Candidate(f"候補者{len(self.party.candidates) + 1}", region_name))
                recruited_info.append(f"{region_name}に{count}名")
            self.party.funds -= total_cost; QMessageBox.information(self, "擁立完了", f"{'、'.join(recruited_info)}を擁立しました！"); self.update_all_displays()
    def open_organize_dialog(self):
        if self.party.funds < 5000: QMessageBox.warning(self, "資金不足", "地域政党の組織には5000万円必要です。"); return
        ally_regions = [ally.specialty['regions'][0] for ally in self.party.regional_allies if ally.specialty.get('type') == 'limited_activity']
        organizable_regions = [r for r in self.regions.keys() if r not in ally_regions]
        if not organizable_regions: QMessageBox.information(self, "組織不可", "全ての選挙区に、すでにあなたの息のかかった地域政党が存在します。"); return
        dialog = OrganizePartyDialog(organizable_regions, self)
        if dialog.exec() and dialog.selected_region: self.execute_organize_party(dialog.selected_region)
    def execute_organize_party(self, region_name):
        self.party.funds -= 5000; party_name = f"{region_name}連合"
        specialty = {'type': 'limited_activity', 'regions': [region_name]}; new_ally = RivalParty(party_name, 'C', specialty, owner=self.party.name)
        self.party.regional_allies.append(new_ally); self.active_rivals.append(new_ally); self.active_parties.append(new_ally.name)
        for r in self.regions.values(): r.support_rates[new_ally.name] = 0
        region = self.regions[region_name]; gain = 5
        other_parties = [p for p in self.active_parties if p != new_ally.name]
        if other_parties:
            total_loss = 0
            for i, p_name in enumerate(other_parties):
                loss = (gain // len(other_parties)) + (1 if i < gain % len(other_parties) else 0)
                actual_loss = min(region.support_rates.get(p_name,0), loss); region.support_rates[p_name] -= actual_loss; total_loss += actual_loss
            region.support_rates[new_ally.name] = total_loss
        else: region.support_rates[new_ally.name] = gain
        current_total = sum(region.support_rates.values())
        if current_total != 100: region.support_rates[self.party.name] += (100 - current_total)
        QMessageBox.information(self, "組織完了", f"{region_name}に、あなたの影響下にある地域政党「{party_name}」が誕生しました！"); self.update_all_displays()
    def _predict_all_seats(self):
        predicted_seats = {p: 0 for p in self.active_parties}
        for region in self.regions.values():
            votes = region.get_votes_per_party(); candidates_in_this_region = {}
            player_candidate_count = sum(1 for c in self.party.candidates if c.assigned_region == region.name)
            candidates_in_this_region[self.party.name] = player_candidate_count
            all_ai_parties = self.active_rivals
            for p in all_ai_parties: candidates_in_this_region[p.name] = region.seats
            allocated_seats = self._calculate_donto(votes, region.seats, candidates_in_this_region)
            for party, seats in allocated_seats.items():
                if party in predicted_seats: predicted_seats[party] += seats
        return predicted_seats
    def open_coalition_dialog(self):
        if self.days_left < 3: QMessageBox.warning(self, "実行不可", "選挙期間が足りません (3日必要)。"); return
        predicted_seats = self._predict_all_seats(); my_seats = predicted_seats.get(self.party.name, 0)
        negotiable_parties_data = []
        all_allied_parties = self.get_all_allied_parties()
        for rival in self.active_rivals:
            if rival.owner == 'cpu' and rival.name not in all_allied_parties:
                rival_seats = predicted_seats.get(rival.name, 0); penalty = 0
                if rival_seats > my_seats:
                    if rival.rank == 'A': penalty = -2
                    elif rival.rank == 'B': penalty = -5
                    else: penalty = -7
                final_prob = max(0, rival.coalition_probability + penalty)
                negotiable_parties_data.append((rival, final_prob, penalty))
        if not negotiable_parties_data: QMessageBox.information(self, "交渉不可", "交渉可能な相手が全員いずれかの連合に所属しています。"); return
        dialog = CoalitionDialog(negotiable_parties_data, self)
        if dialog.exec() and dialog.selected_rival: self.execute_coalition_negotiation(dialog.selected_rival)
    def execute_coalition_negotiation(self, rival):
        for i in range(3):
            if self.days_left <= 0: break
            turn_feedback_message = f"「{rival.name}」との交渉継続中… ({i+1}/3日目)"; self.end_of_day_processing(0, turn_feedback_message)
        if self.days_left <= 0: return
        predicted_seats = self._predict_all_seats(); my_seats = predicted_seats.get(self.party.name, 0); rival_seats = predicted_seats.get(rival.name, 0)
        penalty = 0
        if rival_seats > my_seats:
            if rival.rank == 'A': penalty = -2
            elif rival.rank == 'B': penalty = -5
            else: penalty = -7
        final_prob = max(0, rival.coalition_probability + penalty)
        success = random.randint(1, 100) <= final_prob
        if success:
            self.party.coalition_partners.append(rival.name); QMessageBox.information(self, "交渉成功", f"「{rival.name}」との連立政権樹立に合意しました！")
        else: QMessageBox.information(self, "交渉決裂", f"「{rival.name}」との交渉は決裂しました…\nしかし、交渉の素地はできたようだ。")
        increase = random.randint(5, 15); rival.coalition_probability = min(100, rival.coalition_probability + increase)
        self.update_all_displays()
    def calculate_daily_income(self):
        total_supporters = 0
        for region in self.regions.values(): total_supporters += region.population * (region.support_rates.get(self.party.name, 0) / 100)
        for ally in self.party.regional_allies:
            for region in self.regions.values(): total_supporters += region.population * (region.support_rates.get(ally.name, 0) / 100)
        income = (int(total_supporters) // 50000) * 10
        return income
    def run_rival_ai(self, rival):
        specialty = rival.specialty; target_region = None
        if random.random() < 0.7:
            if specialty['type'] == 'policy': target_region = max(self.regions.values(), key=lambda r: r.interests.get(specialty['category'], 0))
            elif specialty['type'] == 'speech': target_region = max(self.regions.values(), key=lambda r: r.age_demographics.get(specialty['theme'], 0))
            elif specialty['type'] == 'region_type':
                candidate_regions = [r for r in self.regions.values() if r.area_type == specialty['area']]
                target_region = random.choice(candidate_regions) if candidate_regions else random.choice(list(self.regions.values()))
            elif specialty['type'] == 'region_specific': target_region = self.regions[specialty['region']]
            elif specialty['type'] == 'limited_activity': target_region = self.regions[random.choice(specialty['regions'])]
            elif specialty['type'] == 'nationwide_ally': target_region = random.choice(list(self.regions.values()))
            elif specialty['type'] == 'high_seats': target_region = max(self.regions.values(), key=lambda r: r.seats)
        if not target_region: target_region = random.choice(list(self.regions.values()))
        gain = random.randint(rival.support_range[0], rival.support_range[1])
        other_parties = [p for p in self.active_parties if p != rival.name]
        if not other_parties: return
        total_gain = 0
        for _ in range(gain):
            eligible_victims = [p for p in other_parties if target_region.support_rates.get(p, 0) > 0]
            if not eligible_victims: break
            victim_party = random.choice(eligible_victims); target_region.support_rates[victim_party] -= 1; total_gain += 1
        if rival.name not in target_region.support_rates: target_region.support_rates[rival.name] = 0
        target_region.support_rates[rival.name] += total_gain
        current_total = sum(target_region.support_rates.values())
        if current_total != 100: target_region.support_rates[rival.name] += (100 - current_total)
    def run_regional_ally_ai(self, ally):
        target_region = self.regions[ally.specialty['regions'][0]]; gain = random.randint(ally.support_range[0], ally.support_range[1])
        other_parties = [p for p in self.active_parties if p != ally.name]
        if not other_parties: return
        victim = random.choice(other_parties); actual_loss = min(target_region.support_rates[victim], gain)
        target_region.support_rates[victim] -= actual_loss; target_region.support_rates[ally.name] += actual_loss
        current_total = sum(target_region.support_rates.values())
        if current_total != 100: target_region.support_rates[ally.name] += (100 - current_total)
    def run_all_ai(self):
        for rival in self.active_rivals[:]: self.run_rival_ai(rival)
        if self.days_left % 2 == 0:
            for ally in [a for a in self.party.regional_allies if a.specialty.get('type') == 'limited_activity']: self.run_regional_ally_ai(ally)
    def run_a_rank_rival_organization_ai(self):
        a_rank_rivals = [r for r in self.active_rivals if r.rank == 'A' and r.owner == 'cpu']
        for rival in a_rank_rivals:
            if random.random() < 0.5: 
                all_regional_party_regions = []
                all_regional_parties = [p for p in self.active_rivals if p.specialty.get('type') in ['limited_activity', 'region_specific']]
                all_regional_parties.extend(self.party.regional_allies)
                for p in all_regional_parties:
                    if 'region' in p.specialty: all_regional_party_regions.append(p.specialty['region'])
                    elif 'regions' in p.specialty: all_regional_party_regions.extend(p.specialty['regions'])
                organizable_regions = [r.name for r in self.regions.values() if r.area_type in ['町', '村'] and r.name not in all_regional_party_regions]
                if not organizable_regions: continue
                target_region_name = random.choice(organizable_regions)
                ally_name_base = f"{rival.name[:2]}地域党"; ally_name = ally_name_base; suffix = 1
                while ally_name in self.active_parties: suffix += 1; ally_name = f"{ally_name_base}{suffix}"
                specialty = {'type': 'limited_activity', 'regions': [target_region_name]}; new_ally = RivalParty(ally_name, 'C', specialty, owner=rival.name)
                self.active_rivals.append(new_ally); self.active_parties.append(new_ally.name)
                for r in self.regions.values(): r.support_rates[new_ally.name] = 0
                region = self.regions[target_region_name]; gain = 5 
                other_parties = [p for p in self.active_parties if p not in [new_ally.name, rival.name]]
                if other_parties:
                    total_loss = 0; loss_per_party = gain // len(other_parties)
                    for i, p_name in enumerate(other_parties):
                        loss = loss_per_party + (1 if i < gain % len(other_parties) else 0)
                        actual_loss = min(region.support_rates.get(p_name, 0), loss); region.support_rates[p_name] -= actual_loss; total_loss += actual_loss
                    ally_gain = round(total_loss * 0.6); parent_gain = total_loss - ally_gain
                    region.support_rates[new_ally.name] += ally_gain; region.support_rates[rival.name] += parent_gain
                current_total = sum(region.support_rates.values())
                if current_total != 100: region.support_rates[rival.name] += (100 - current_total)
                self.app_instance.statusBar().showMessage(f"【速報】「{rival.name}」が{target_region_name}に地域政党「{ally_name}」を組織！", 6000)
    def get_all_allied_parties(self):
        allied_parties = set()
        if self.party:
            allied_parties.add(self.party.name); allied_parties.update(self.party.coalition_partners); allied_parties.update([ally.name for ally in self.party.regional_allies])
        for coalition in self.ai_coalitions: allied_parties.update(coalition)
        return allied_parties
    def run_ai_coalition_logic(self):
        if random.random() > 0.03: return
        predicted_seats = self._predict_all_seats(); all_allied_parties = self.get_all_allied_parties()
        candidates = [p for p in self.active_rivals if p.owner == 'cpu' and p.name not in all_allied_parties and predicted_seats.get(p.name, 0) <= 3]
        if len(candidates) < 2: return
        party1, party2 = random.sample(candidates, 2)
        new_coalition = {party1.name, party2.name}; self.ai_coalitions.append(new_coalition)
        msg = f"【政界再編】「{party1.name}」と「{party2.name}」が新たな連合を結成！"; self.app_instance.statusBar().showMessage(msg, 8000); self.update_all_displays()
    def execute_turn(self):
        action_name_raw = self.action_combo.currentText(); action_name = action_name_raw.split(' (')[0]; cost = self.actions.get(action_name, 0)
        if self.party.funds < cost: QMessageBox.warning(self, "資金不足", f"資金が足りません！ (必要: {cost}万円)"); return
        if action_name == "テレビ演説" and self.tv_speech_left <= 0: QMessageBox.warning(self, "実行不可", "テレビ演説はもう行えません。"); return
        turn_feedback_message = ""
        def apply_support_gain(region, gain, party_name):
            other_parties = [p for p in self.active_parties if p != party_name];
            if not other_parties: return
            for _ in range(gain):
                eligible_victims = [p for p in other_parties if region.support_rates.get(p, 0) > 0]
                if not eligible_victims: break
                victim_party = random.choice(eligible_victims)
                region.support_rates[victim_party] -= 1; region.support_rates[party_name] += 1
            current_total = sum(region.support_rates.values())
            if current_total != 100: region.support_rates[party_name] += (100 - current_total)
        if action_name == "テレビ演説":
            self.tv_speech_left -= 1; detail_name = self.detail_combo.currentText(); policy = next(p for p in self.policies if p.name == detail_name)
            base_effect = random.uniform(0.3, 1.2)
            if base_effect >= 1.0: turn_feedback_message = "テレビ演説は素晴らしい反響を呼び、大成功だった！"
            elif base_effect >= 0.6: turn_feedback_message = "テレビ演説は、まずまずの反応を得た。"
            else: turn_feedback_message = "残念ながら、テレビ演説の反応は今ひとつだったようだ…"
            for region in self.regions.values():
                interest = region.interests.get(policy.category, 0.1); support_gain = int(30 * base_effect * interest)
                if support_gain > 0: apply_support_gain(region, support_gain, self.party.name)
        elif action_name == "何もしない":
            recruited_regions = list(set(c.assigned_region for c in self.party.candidates))
            if not recruited_regions: QMessageBox.warning(self, "実行不可", "候補者がいないため「何もしない」もできません。"); return
            target_region_name = random.choice(recruited_regions); region = self.regions[target_region_name]
            apply_support_gain(region, 1, self.party.name)
            turn_feedback_message = f"{target_region_name}で世論の追い風が吹き、支持がわずかに上昇した。"
        else:
            region_name_with_count = self.region_combo.currentText()
            if not region_name_with_count: QMessageBox.warning(self, "実行不可", "行動対象の選挙区がありません。"); return
            region_name = region_name_with_count.split(' (')[0]; detail_name = self.detail_combo.currentText()
            turn_feedback_message = f"{region_name}で「{detail_name}」に関する「{action_name}」を行った。"
            region = self.regions[region_name]; base_effect = {"政策発表": 1.0, "街頭演説": 0.6, "チラシ配り": 0.3}[action_name]; support_gain = 0
            if action_name in ["政策発表", "チラシ配り"]:
                policy = next(p for p in self.policies if p.name == detail_name); interest = region.interests.get(policy.category, 0.1); support_gain = int(30 * base_effect * interest)
            elif action_name == "街頭演説":
                age_map = {"若者向け": "youth", "現役世代向け": "middle", "高齢者向け": "senior"}
                target_key = age_map[detail_name.replace("演説","")]; age_ratio = region.age_demographics.get(target_key, 0.1); support_gain = int(30 * base_effect * age_ratio)
            if support_gain > 0: apply_support_gain(region, support_gain, self.party.name)
        self.end_of_day_processing(cost, turn_feedback_message)
    def end_of_day_processing(self, cost, feedback_message):
        if self.days_left <= 0: return
        self.party.funds -= cost; self.days_left -= 1
        daily_income = self.calculate_daily_income(); self.party.funds += daily_income
        feedback_with_income = f"{feedback_message} | 本日の収入: {daily_income}万円"
        self.run_all_ai()
        elapsed_days = 65 - self.days_left
        if elapsed_days > 0 and elapsed_days % 20 == 0: self.run_a_rank_rival_organization_ai()
        if self.days_left > 0 and self.days_left % 5 == 0: self.run_ai_coalition_logic()
        self.update_all_displays()
        if feedback_with_income: self.app_instance.statusBar().showMessage(feedback_with_income, 5000)
        if self.days_left == 35: self.show_report("中間発表")
        elif self.days_left <= 0: self.show_report("最終結果")
    def _calculate_donto(self, votes_per_party, seats, candidates_per_party):
        quotients = []
        for party, votes in votes_per_party.items():
            if votes > 0:
                max_seats_for_party = candidates_per_party.get(party, seats)
                for i in range(1, max_seats_for_party + 1): quotients.append({'party': party, 'q': votes / i})
        sorted_quotients = sorted(quotients, key=lambda x: x['q'], reverse=True)
        allocated_seats = {party: 0 for party in votes_per_party.keys()}
        for i in range(seats):
            if i < len(sorted_quotients):
                winner = sorted_quotients[i]['party']; allocated_seats[winner] += 1
        return allocated_seats
    def create_pie_chart(self, final_seats, player_coalition_names, ai_coalitions):
        try: plt.rcParams['font.family'] = 'Yu Gothic'
        except:
            try: plt.rcParams['font.family'] = 'Hiragino Sans'
            except:
                try: plt.rcParams['font.family'] = 'IPAexGothic'
                except: pass
        chart_data = {}; processed_parties = set(); colors_map = self.get_party_colors_map()
        player_total_seats = sum(final_seats.get(p, 0) for p in player_coalition_names)
        if player_total_seats > 0: chart_data[f"{self.party.name} 連合"] = player_total_seats
        processed_parties.update(player_coalition_names)
        for i, coalition in enumerate(ai_coalitions):
            coalition_total_seats = sum(final_seats.get(p, 0) for p in coalition)
            if coalition_total_seats > 0:
                representative = list(coalition)[0]; chart_data[f"{representative} 連合"] = coalition_total_seats
            processed_parties.update(coalition)
        for party_name, seats in final_seats.items():
            if party_name not in processed_parties and seats > 0: chart_data[party_name] = seats
        sorted_chart_data = dict(sorted(chart_data.items(), key=lambda item: item[1], reverse=True))
        if not sorted_chart_data: return None
        pie_labels = list(sorted_chart_data.keys()); pie_sizes = list(sorted_chart_data.values()); pie_colors = []
        for label in pie_labels:
            original_party_name = label.split(' ')[0]; pie_colors.append(colors_map.get(original_party_name, 'gray'))
        fig, ax = plt.subplots(figsize=(8, 5.5), dpi=100)
        wedges, texts, autotexts = ax.pie(pie_sizes, autopct='%1.1f%%', startangle=90, colors=pie_colors, pctdistance=0.8, wedgeprops={'linewidth': 1, 'edgecolor':'white'})
        plt.setp(autotexts, size=8, weight="bold", color="white")
        ax.axis('equal')
        font_properties = {'weight' : 'bold', 'size': 9}
        legend_labels = [f"{label} ({sorted_chart_data[label]}議席)" for label in pie_labels]
        ax.legend(wedges, legend_labels, title="政党 / 連合", loc="center left", bbox_to_anchor=(1, 0, 0.5, 1), prop=font_properties)
        plt.title("最終議席配分", pad=20, loc='center'); fig.tight_layout()
        chart_path = "election_results_pie.png"; plt.savefig(chart_path); plt.close(fig)
        return chart_path
    def show_report(self, title):
        report_text = f"--- {title} ---\n\n"; final_seats = {p: 0 for p in self.active_parties}
        player_coalition_names = [self.party.name]; player_coalition_names.extend(self.party.coalition_partners); player_coalition_names.extend([ally.name for ally in self.party.regional_allies])
        all_coalitions = set(player_coalition_names)
        for ai_c in self.ai_coalitions: all_coalitions.update(ai_c)
        for region in self.regions.values():
            report_text += f"▼ {region.name} (定数: {region.seats})\n"; all_votes_in_region = region.get_votes_per_party()
            eligible_votes = {}
            for party_name, votes in all_votes_in_region.items():
                is_player_candidate = self.party.name == party_name and any(c.assigned_region == region.name for c in self.party.candidates)
                is_rival = party_name in [r.name for r in self.active_rivals]
                if is_player_candidate or is_rival:
                    eligible_votes[party_name] = votes
            for party, v in sorted(all_votes_in_region.items(), key=lambda i: i[1], reverse=True):
                eligibility_mark = "" if party in eligible_votes else " (議席獲得資格なし)"; coalition_mark = " (連合)" if party in all_coalitions else ""
                report_text += f"  ・{party}: {v:,}票 ({region.support_rates.get(party, 0)}%){coalition_mark}{eligibility_mark}\n"
            candidates_in_this_region = {}
            player_candidate_count = sum(1 for c in self.party.candidates if c.assigned_region == region.name)
            candidates_in_this_region[self.party.name] = player_candidate_count
            for p in self.active_rivals: candidates_in_this_region[p.name] = region.seats
            allocated_seats = self._calculate_donto(eligible_votes, region.seats, candidates_in_this_region)
            report_text += "  [議席配分]\n"
            if sum(allocated_seats.values()) == 0: report_text += "    (議席獲得なし)\n"
            else:
                for party, s in allocated_seats.items():
                    if s > 0:
                        coalition_mark = " (連合)" if party in all_coalitions else ""; report_text += f"    {party}: {s}議席{coalition_mark}\n"
                        if party in final_seats: final_seats[party] += s
            report_text += "\n"
        if title == "最終結果":
            self.execute_button.setEnabled(False); self.recruit_button.setEnabled(False); self.coalition_button.setEnabled(False); self.organize_button.setEnabled(False); self.found_ally_button.setEnabled(False)
            my_seats = final_seats.get(self.party.name, 0); player_ally_seats = sum(final_seats.get(ally.name, 0) for ally in self.party.regional_allies)
            coalition_seats = my_seats + player_ally_seats + sum(final_seats.get(p, 0) for p in self.party.coalition_partners)
            summary = f"--- 選挙結果 (総議席: {self.total_seats} / 過半数: {self.majority_threshold}議席) ---\n\n"; summary += f"あなたの党議席      : {my_seats}議席\n"; summary += f"系列・地域政党との合計 : {my_seats + player_ally_seats}議席\n"
            if self.party.coalition_partners: summary += f"連合全体の合計議席    : {coalition_seats}議席\n"
            summary += f"最終資金              : {self.party.funds:,}万円\n\n"
            player_bloc_for_rival_check = player_coalition_names; rival_seats_list = [s for p, s in final_seats.items() if p not in player_bloc_for_rival_check]; max_rival_seats = max(rival_seats_list) if rival_seats_list else 0
            victory_achieved = False; is_gekokujo_coalition = False
            if self.party.coalition_partners:
                if all(r.rank == 'C' for p_name in self.party.coalition_partners for r in self.active_rivals if r.name == p_name): is_gekokujo_coalition = True
            if (my_seats + player_ally_seats) >= self.majority_threshold: summary += f"【★★★★★ 完全勝利】\nおめでとうございます！\nあなたの党は、系列・地域政党と合わせて{my_seats + player_ally_seats}議席を獲得し、見事単独過半数を達成しました！"; victory_achieved = True
            elif is_gekokujo_coalition and coalition_seats >= self.majority_threshold: summary += f"【★★★★★ 下剋上達成！】\n格下のCクラス政党のみをまとめ上げ、見事、過半数を獲得！\nまさに下剋上と言える、歴史的な勝利です！"; victory_achieved = True
            elif coalition_seats >= self.majority_threshold:
                partner_names = "と".join(self.party.coalition_partners); victory_message = "【★★★★☆ 連立政権樹立】\n"
                if partner_names: victory_message += f"「{partner_names}」との連立"
                if self.party.regional_allies:
                    if partner_names: victory_message += "、そして"
                    victory_message += "系列・地域政党との協力"
                victory_message += f"により、合計{coalition_seats}議席で過半数を獲得。政権を樹立しました！"; summary += victory_message; victory_achieved = True
            elif coalition_seats > max_rival_seats: summary += f"【★★★☆☆ 比較第一党】\n過半数には届きませんでしたが、あなたの率いる連合は{coalition_seats}議席を獲得。\n他のいかなる勢力も上回る、比較第一党の地位を確保しました。\n今後の政局の主導権を握ることになるでしょう。"; victory_achieved = True
            elif len(self.party.regional_allies) >= 6: summary += f"【★★★☆☆ 草の根の勝利】\n過半数獲得はなりませんでしたが、{len(self.party.regional_allies)}の地域・系列政党を組織し、全国に強力な地盤を築きました。\nあなたの影響力は、議席数だけでは測れないものとなっています。"; victory_achieved = True
            if self.party.funds >= 200000: summary += f"\n\n【☆☆☆☆☆ 金権政治の完成】\n議席数では他を圧倒できなかったかもしれません。\nしかし、{self.party.funds:,}万円という莫大な資金を手にしました。\nもはやあなたの党は、日本の政治を裏から操る強大な存在です。"; victory_achieved = True 
            if not victory_achieved: summary += "【★☆☆☆☆ 敗北】\nいずれの枠組みでも過半数に届かず、政権樹立はなりませんでした。"
            report_text += "----------------------------------------\n" + summary
            chart_path = self.create_pie_chart(final_seats, player_coalition_names, self.ai_coalitions)
            dialog = ResultDialog(title, report_text, chart_path, self)
            dialog.exec()
            reply = QMessageBox.question(self, "ゲーム終了", "もう一度遊びますか？", QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No, QMessageBox.StandardButton.No)
            if reply == QMessageBox.StandardButton.Yes: self.start_game(); self.execute_button.setEnabled(True); self.recruit_button.setEnabled(True); self.coalition_button.setEnabled(True); self.organize_button.setEnabled(True); self.found_ally_button.setEnabled(True)
            else: self.app_instance.show_title_screen()
        else:
            dialog = ResultDialog(title, report_text, None, self)
            dialog.exec()
    def update_all_displays(self):
        if not self.party: return # ゲーム開始前は何もしない
        seats_r = len(self.party.candidates); coalition_status = f" | 連立: {', '.join(self.party.coalition_partners)}" if self.party.coalition_partners else ""; regional_ally_status = f" | 系列・地域政党: {len(self.party.regional_allies)}党" if self.party.regional_allies else ""
        self.status_label.setText(f"党資金: {self.party.funds:,}万円 | 残り{self.days_left}日 | 擁立数: {seats_r}人{coalition_status}{regional_ally_status}")
        self.region_combo.blockSignals(True); current_selection_text = self.region_combo.currentText(); self.region_combo.clear()
        candidate_counts = Counter(c.assigned_region for c in self.party.candidates); recruited_regions_with_counts = []
        for region_name in self.regions.keys():
            count = candidate_counts.get(region_name, 0)
            if count > 0: recruited_regions_with_counts.append(f"{region_name} ({count}人)")
        if recruited_regions_with_counts:
            self.region_combo.addItems(recruited_regions_with_counts)
            if current_selection_text in recruited_regions_with_counts: self.region_combo.setCurrentText(current_selection_text)
            else: self.region_combo.setCurrentIndex(0)
        self.region_combo.blockSignals(False); can_act = self.days_left > 0
        self.execute_button.setEnabled(can_act); self.recruit_button.setEnabled(can_act); self.coalition_button.setEnabled(can_act); self.organize_button.setEnabled(can_act); self.found_ally_button.setEnabled(can_act)
        self.update_action_combo(); self.update_detail_combo(); self.update_info_display_on_demand()
        if self.party:
            all_allied_parties = self.get_all_allied_parties()
            party_colors_map = self.get_party_colors_map()
            self.map_widget.update_map(self.regions, party_colors_map, all_allied_parties)
    def update_action_combo(self):
        self.action_combo.blockSignals(True); current_action_text = self.action_combo.currentText(); self.action_combo.clear(); action_items = []
        for action, cost in self.actions.items():
            cost_text = f" ({cost}万)" if cost > 0 else ""
            if action == "テレビ演説": action_items.append(f"テレビ演説 (残り{self.tv_speech_left}回){cost_text}")
            else: action_items.append(f"{action}{cost_text}")
        self.action_combo.addItems(action_items)
        if current_action_text in action_items: self.action_combo.setCurrentText(current_action_text)
        self.action_combo.blockSignals(False)
    def update_info_display_on_demand(self):
        region_name_with_count = self.region_combo.currentText()
        if not region_name_with_count or not self.party: self.info_display.setText("候補者を擁立すると、選挙区の情報が表示されます。\n右上の「マップ」タブで勢力図を確認できます。"); return
        region_name = region_name_with_count.split(' (')[0]
        region = self.regions[region_name]; html = f"<h3>選挙区: {region.name} (定数: {region.seats})</h3>"
        candidates_in_region = [c.name for c in self.party.candidates if c.assigned_region == region_name]
        html += f"<h4>あなたの党の状況</h4><p><b>擁立候補者:</b> {', '.join(candidates_in_region) if candidates_in_region else 'なし'} (計{len(candidates_in_region)}名)</p>"
        html += "<h4>地域特性</h4><ul>"
        for topic, val in {**region.interests, **region.age_demographics}.items():
            topic_jp=topic.replace("youth","若者").replace("middle","現役").replace("senior","高齢者"); html += f"<li>{topic_jp}関連の関心度: {int(val*100)}%</li>"
        html += "</ul><h4>現在の支持率</h4><ul>"
        all_allied_parties = self.get_all_allied_parties()
        for party, rate in sorted(region.support_rates.items(), key=lambda i: i[1], reverse=True):
            coalition_mark = " <span style='color: gold; font-weight:bold;'>(連合)</span>" if party in all_allied_parties else ""
            html += f"<li><b>{party}:</b> {rate}%{coalition_mark}</li>"
        html += "</ul>"; self.info_display.setHtml(html)
    def update_detail_combo(self):
        action_raw = self.action_combo.currentText(); action = action_raw.split(' (')[0]
        self.detail_combo.clear(); is_region_specific = action not in ["何もしない", "テレビ演説"]; self.region_combo.setEnabled(is_region_specific)
        has_details = action not in ["何もしない"]; self.detail_combo.setEnabled(has_details)
        if action in ["政策発表", "チラシ配り", "テレビ演説"]: self.detail_combo.addItems([p.name for p in self.policies])
        elif action == "街頭演説": self.detail_combo.addItems(self.speech_themes)

class ElectionSimApp(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Game of Civic Strategy"); self.resize(1200, 800)
        self.init_pygame()
        self.current_bgm_path = None
        self.init_ui()
    def init_pygame(self):
        if PYGAME_AVAILABLE:
            try: pygame.mixer.init(); print("[DEBUG] Pygameミキサーが初期化されました。")
            except pygame.error as e: print(f"[エラー] Pygameミキサーの初期化エラー: {e}")
    def init_ui(self):
        self.stacked_widget = QStackedWidget(); self.setCentralWidget(self.stacked_widget)
        self.title_screen = TitleScreen(self); self.game_screen = GameScreen(self); self.credits_screen = CreditsScreen(self)
        self.stacked_widget.addWidget(self.title_screen); self.stacked_widget.addWidget(self.game_screen); self.stacked_widget.addWidget(self.credits_screen)
        
        status_bar = QStatusBar()
        self.setStatusBar(status_bar)
        self.volume_slider = QSlider(Qt.Orientation.Horizontal)
        self.volume_slider.setRange(0, 100); self.volume_slider.setValue(70); self.volume_slider.setFixedWidth(120)
        self.volume_slider.valueChanged.connect(self.set_volume)
        status_bar.addPermanentWidget(QLabel(" BGM音量: ")); status_bar.addPermanentWidget(self.volume_slider)

        self.show_title_screen()
        if PYGAME_AVAILABLE: QApplication.instance().aboutToQuit.connect(pygame.mixer.quit)
        
    def set_volume(self, value):
        if PYGAME_AVAILABLE: pygame.mixer.music.set_volume(value / 100.0)

    def show_title_screen(self):
        self.stacked_widget.setCurrentWidget(self.title_screen); self.play_music(TITLE_BGM_PATH)
    def show_game_screen(self):
        self.stacked_widget.setCurrentWidget(self.game_screen); self.game_screen.start_game(); self.play_music(GAME_BGM_PATH)
    def show_credits_screen(self):
        self.stacked_widget.setCurrentWidget(self.credits_screen); self.credits_screen.start_scrolling(); self.play_music(CREDITS_BGM_PATH)
    
    def play_music(self, music_path):
        if not PYGAME_AVAILABLE: return
        if self.current_bgm_path == music_path and pygame.mixer.music.get_busy(): return
        if not os.path.exists(music_path): print(f"[警告] BGMファイルが見つかりません: {music_path}"); pygame.mixer.music.stop(); self.current_bgm_path = None; return
        try:
            pygame.mixer.music.load(music_path); pygame.mixer.music.set_volume(self.volume_slider.value() / 100.0)
            pygame.mixer.music.play(-1); self.current_bgm_path = music_path
        except pygame.error as e: print(f"[エラー] BGMの読み込み/再生エラー: {e}")

if __name__ == '__main__':
    # アセット用のディレクトリが存在しない場合に作成
    os.makedirs(os.path.join(ASSET_DIR, "bgm"), exist_ok=True)
    os.makedirs(os.path.join(ASSET_DIR, "titlepng"), exist_ok=True)
    
    app = QApplication(sys.argv)
    if os.path.exists(APP_ICON_PATH): app.setWindowIcon(QIcon(APP_ICON_PATH))
    main_window = ElectionSimApp()
    main_window.show()
    sys.exit(app.exec())